# -*- coding: utf-8 -*-

import re
import json
import logging
from functools import wraps
from datetime import datetime, date, timedelta, time

from django.conf import settings
from django.http import HttpResponseBadRequest, HttpResponse
from django.utils.html import escape
from django.utils.encoding import smart_unicode, smart_str

from common.xgettext.i18n import gettext

from .utils.dict2xml import dict2xml
from .api_errors import ApiError, ApiInternalError

log = logging.getLogger(__name__)

VALID_CALLBACK_RE = re.compile(r'^[$A-Za-z\d_]*$')


def build_response(request, result, http_code):
    out_format = request.GET.get('format', 'json')
    callback = request.GET.get('callback', u'').strip()

    if settings.DEBUG:
        json_str = json.dumps(result, ensure_ascii=False, indent=2, separators=(',', ': '),
                              default=serialize, sort_keys=True)
    else:
        json_str = json.dumps(result, ensure_ascii=False, default=serialize,
                              separators=(',', ':'))

    if out_format == 'json':
        return jsonp_wrap(json_str, callback, http_code)

    elif out_format == 'xml':
        xml_str = u'<?xml version="1.0" encoding="utf-8"?>\n'
        xml_str += dict2xml(dict(response=json.loads(json_str)))

        response = HttpResponse(xml_str.encode('utf-8', 'ignore'),
                                content_type='text/xml; charset=utf-8', status=http_code)
        response['X-Content-Type-Options'] = 'nosniff'
        return response

    else:
        response = HttpResponseBadRequest('Invalid format "{}"'.format(smart_str(escape(out_format))),
                                          content_type='text/plain; charset=utf-8')
        response['X-Content-Type-Options'] = 'nosniff'
        return response


def api_response(view):
    @wraps(view)
    def wrapper(request, *args, **kwargs):
        unexpected_error = gettext(u'Сервис Яндекс.Расписания временно недоступен.'
                                   u' Мы уже в курсе данной проблемы и работаем над ее устранением.'
                                   u' Приносим свои извинения за неудобства.')

        try:
            response = try_build_response(request, view, args, kwargs)
        except Exception as ex:
            if settings.DEBUG:
                raise

            log.error(
                u'Unknown error: %s\n%s\n', repr(ex), smart_unicode(repr(request)),
                exc_info=True,
                extra={'request': request}  # for raven
            )

            result = {'error': {'text': unexpected_error}}
            response = build_response(request, result, 500)

        return response

    return wrapper


def try_build_response(request, view, args, kwargs):
    http_code = 200
    try:
        result = view(request, *args, **kwargs)
    except ApiError as e:
        try:
            version = request.path.split('/')[1]
        except IndexError:
            version = None

        http_code = e.http_code

        result = {'error': {
            'text': e.text,
            'http_code': e.http_code,
            'error_code': u'{}_{}_{}'.format(version, settings.PKG_VERSION, e.error_code),
            'request': request.build_absolute_uri(),
        }}

    if not isinstance(result, dict):
        raise ApiInternalError(u'Неправильный тип ответа: {}\n{}\n'.format(
            smart_unicode(repr(result)),
            smart_unicode(repr(request))
        ))

    return build_response(request, result, http_code)


def jsonp_wrap(json_str, callback, http_code):
    if not VALID_CALLBACK_RE.match(callback):
        response = HttpResponseBadRequest('Invalid callback name "{}"'.format(smart_str(callback)),
                                          content_type='text/plain; charset=utf-8')

        response['X-Content-Type-Options'] = 'nosniff'
        return response

    if callback:
        # Для json_p всегда посылаем 200
        http_code = 200
        response = HttpResponse('{}({});'.format(smart_str(callback), smart_str(json_str)),
                                content_type='text/javascript; charset=utf-8', status=http_code)
        response['X-Content-Type-Options'] = 'nosniff'
        return response
    else:
        response = HttpResponse(smart_str(json_str), content_type="application/json; charset=utf-8",
                                status=http_code)
        response['X-Content-Type-Options'] = 'nosniff'
        return response


def serialize(obj):
    if hasattr(obj, '__json__'):
        return obj.__json__()

    if isinstance(obj, datetime):
        return obj.strftime("%Y-%m-%d %H:%M:%S")

    if isinstance(obj, date):
        return obj.strftime("%Y-%m-%d")

    if isinstance(obj, timedelta):
        return obj.seconds + obj.days * 84600

    if isinstance(obj, time):
        return obj.strftime('%H:%M')

    raise TypeError(repr(obj) + " is not JSON serializable")
