# -*- coding: utf-8 -*-
from datetime import datetime, timedelta, time
from dateutil.relativedelta import relativedelta

from django.conf import settings
from django.core.paginator import Paginator, PageNotAnInteger, EmptyPage
from django.db.models import Q
from django.utils import translation

from common.models.currency import Currency
from common.models.geo import Settlement, Station, StationType
from common.models.schedule import Supplier
from common.models.transport import TransportType
from travel.rasp.library.python.common23.date import environment
from common.views.currency import CurrencyInfo
from common.xgettext.i18n import gettext
from route_search.shortcuts import find, find_on_all_days, PlainSegmentSearch

from travel.rasp.api_public.api_public.geobase_helpers import calc_points_distance_km
from travel.rasp.api_public.api_public.old_versions.core import points
from travel.rasp.api_public.api_public.old_versions.core.api_errors import ApiError
from travel.rasp.api_public.api_public.old_versions.core.decorators import api_view_method


UP_TO_HOURS_MIN = 24
UP_TO_HOURS_MAX = 24 * 3


class SearchQuery(object):
    pass


class SegmentsSearcher(object):
    def __init__(self, request):
        self.request = request
        self.query = build_search_query(request)
        self.segments = []
        self.search_on_day = bool(self.query.date)

        ids_of_excluded_suppliers = Supplier.objects.filter(exclude_from_external_api=True).values_list('id', flat=True)
        self.threads_filter = (
            ~Q(supplier_id__in=ids_of_excluded_suppliers) &
            ~Q(route__supplier_id__in=ids_of_excluded_suppliers)
        )

    def find_on_date(self):
        query = self.query
        start_dt = query.point_from.pytz.localize(datetime.combine(query.date, time(0, 0)))
        end_dt = start_dt + timedelta(hours=query.up_to_hours)

        local_start_date = start_dt.date()
        raw_segments = find(
            query.point_from, query.point_to, local_start_date, query.t_types,
            threads_filter=self.threads_filter,
        )

        self.segments = []
        for segment in raw_segments:
            if segment.departure < start_dt:
                continue

            if segment.departure > end_dt:
                break

            self.segments.append(segment)

        return self.segments

    def find_on_all_days(self):
        self.segments = find_on_all_days(
            self.query.point_from, self.query.point_to, self.query.t_types,
            threads_filter=self.threads_filter,
        )

        return self.segments

    def find_on_all_days_grouped(self, title_grouping=True):
        search = PlainSegmentSearch(
            self.query.point_from, self.query.point_to, self.query.t_types,
            threads_filter=self.threads_filter,
        )
        segment_groups = search.all_days_group(title_grouping=title_grouping)

        return segment_groups

    def search(self):
        if self.search_on_day:
            return self.find_on_date()

        return self.find_on_all_days()


def same_suburban_zone(point_from, point_to):
    for d1 in point_from.get_externaldirections():
        for d2 in point_to.get_externaldirections():
            if d1.suburban_zone_id == d2.suburban_zone_id is not None:
                return True

    return False


def get_transport_types(request):
    t_types_param = request.GET.get(
        'transport_types', request.GET.get('transport_type', u'')
    ).strip()

    t_type_codes = {s for s in map(unicode.strip, t_types_param.split(u',')) if s}

    if not t_type_codes or 'all' in t_type_codes:
        return

    if len(t_type_codes) > 10:
        raise ApiError(gettext(u'Неверный формат запроса {}').format(
            t_types_param
        ), http_code=400)

    water_type_codes = {'water', 'river', 'sea'}
    deprecated_water_types = ['river', 'sea']

    if any(water_type_code in t_type_codes for water_type_code in water_type_codes):
        t_type_codes |= water_type_codes

    t_types = []
    for t_type_code in t_type_codes:
        try:
            t_types.append(TransportType.objects.get(code=t_type_code))
        except TransportType.DoesNotExist:
            if t_type_code in deprecated_water_types:
                continue
            raise ApiError(gettext(u'Неподдерживаемый тип транспорта {}').format(
                t_type_code
            ), http_code=400)

    return t_types


def get_station_types(request):
    station_type_param = request.GET.get('station_type', u'').strip()
    lang = translation.get_language()
    if lang not in settings.MODEL_LANGUAGES:
        lang = 'ru'

    station_type_names = [s for s in map(unicode.strip, station_type_param.split(u',')) if s]

    if not station_type_names or 'all' in station_type_names:
        return

    if len(station_type_names) > 10:
        raise ApiError(gettext(u'Не верный формат запроса {}').format(
            station_type_names
        ), http_code=400)

    station_types = []

    for station_type_name in station_type_names:
        try:
            station_types.append(StationType.objects.get(
                **{'name_{}'.format(lang): station_type_name}
            ))
        except StationType.DoesNotExist:
            raise ApiError(gettext(u'Не поддерживаемый тип станции {}').format(
                station_type_name
            ), http_code=400)

    return station_types


def build_search_query(request):
    query = SearchQuery()

    system_from = system_to = request.GET.get('system', u'yandex')

    system_from = request.GET.get('system_from') or system_from
    system_to = request.GET.get('system_to') or system_to

    code_from = request.GET.get('from', u'').strip()
    code_to = request.GET.get('to', u'').strip()

    query.date = get_date_from_request(request)

    if not (code_from and code_to):
        raise ApiError(gettext(u'Нужно указать from и to'), http_code=400)

    query.point_from = points.PointFinder.find_point(system_from, code_from)
    query.point_to = points.PointFinder.find_point(system_to, code_to)

    if query.point_to == query.point_from:
        raise ApiError(gettext(u'Укажите пожалуйста разные точки отправления и прибытия'),
                       http_code=400)

    if query.date:
        up_to_is_good = True
        try:
            query.up_to_hours = float(request.GET.get('up_to', u'24'))
        except ValueError:
            up_to_is_good = False

        if query.up_to_hours < UP_TO_HOURS_MIN or query.up_to_hours > UP_TO_HOURS_MAX:
            up_to_is_good = False

        if not up_to_is_good:
            raise ApiError(
                gettext(u'Параметр up_to должен быть'
                        u' положительным числом в интервале {} {},'
                        u' по умолчанию 24')
                .format(
                    UP_TO_HOURS_MIN, UP_TO_HOURS_MAX
                ), http_code=400)

    query.t_types = get_transport_types(request)

    return query


def check_date_range(date):
    start_date = (environment.now_aware() - relativedelta(days=30)).date()
    end_date = (environment.now_aware() + relativedelta(months=11)).date()

    if not (start_date <= date <= end_date):
        raise ApiError(gettext(u'Указана недопустимая дата - {}. Доступен выбор даты на 30 дней назад и '
                               u'11 месяцев вперед от текущей даты'.format(date)), http_code=400)


def get_date_from_request(request, name='date', default=None):
    date_ = request.GET.get(name, u'').strip()

    if date_:
        try:
            date = datetime.strptime(date_, '%Y-%m-%d').date()

        except ValueError:
            raise ApiError(gettext(u'Дата должна быть в формате YYYY-MM-DD'), http_code=400)

        check_date_range(date)
        return date

    return default


class ApiPaginator(Paginator):
    def validate_number(self, number):
        try:
            super(ApiPaginator, self).validate_number(number)
        except EmptyPage:
            pass

        return int(number)


class ApiPaginateView(object):
    ITEMS_PER_PAGE = 100

    def __init__(self):
        self.request = None
        self.__name__ = self.__class__.__name__  # for opentracing

    @api_view_method
    def __call__(self, request, *args, **kwargs):
        self.request = request
        return self.handle(*args, **kwargs)

    def handle(self, *args, **kwargs):
        raise NotImplementedError()

    def paginate(self, object_list):
        paginator = ApiPaginator(object_list, self.ITEMS_PER_PAGE)

        try:
            page = paginator.page(self.request.GET.get('page', 1))
        except PageNotAnInteger:
            raise ApiError(gettext(u'Неверный номер страницы'), http_code=404)

        result_object_list = page.object_list

        return self.get_pagination_json(page, paginator), result_object_list

    def get_pagination_json(self, page, paginator):
        return {
            'total': paginator.count,
            'page': page.number,
            'has_next': page.has_next(),
            'per_page': paginator.per_page,
            'page_count': paginator.num_pages
        }


def get_settlement_within(center_point, step, distance, collected_settlements):
    min_lat = center_point.latitude - step
    max_lat = center_point.latitude + step
    min_lng = center_point.longitude - step
    max_lng = center_point.longitude + step

    settlement_coords_filter = (
        Q(latitude__isnull=False, longitude__isnull=False) &
        Q(latitude__gte=min_lat, latitude__lte=max_lat) &
        Q(longitude__gte=min_lng, longitude__lte=max_lng)
    )

    main_settlement_station_coords_filter = (
        Q(station__majority=1) &  # Берем одну из главных
        Q(station__latitude__isnull=False, station__longitude__isnull=False) &
        Q(station__latitude__gte=min_lat, station__latitude__lte=max_lat) &
        Q(station__longitude__gte=min_lng, station__longitude__lte=max_lng)
    )

    settlements = list(Settlement.hidden_manager.filter(
        settlement_coords_filter | main_settlement_station_coords_filter
    ).order_by().extra({'s_latitude': 'www_station.latitude',
                        's_longitude': 'www_station.longitude'}))

    for stt in settlements:
        stt.latitude = stt.latitude if stt.latitude is not None else stt.s_latitude
        stt.longitude = stt.longitude if stt.longitude is not None else stt.s_longitude
        stt.distance = calc_points_distance_km(center_point, stt)

    settlements.sort(key=lambda _s: (_s.majority_id, _s.distance))

    result_settlements = []
    for stt in settlements:
        if stt.distance > distance:
            continue

        # Если город уже добавлен, не делаем этого еще раз
        if stt in collected_settlements:
            continue

        collected_settlements.add(stt)
        result_settlements.append(stt)

    return result_settlements


def get_stations_within(center_point, step, distance, collected_stations):
    min_lat = center_point.latitude - step
    max_lat = center_point.latitude + step
    min_lng = center_point.longitude - step
    max_lng = center_point.longitude + step

    coords_filter = (
        Q(latitude__isnull=False, longitude__isnull=False) &
        Q(latitude__gte=min_lat, latitude__lte=max_lat) &
        Q(longitude__gte=min_lng, longitude__lte=max_lng)
    )

    stations = list(Station.hidden_manager.filter(coords_filter).order_by())

    for s in stations:
        s.distance = calc_points_distance_km(center_point, s)

    stations.sort(key=lambda _s: (_s.majority_id, _s.distance))

    result_stations = []
    for s in stations:
        if s.distance > distance:
            continue

        # Если станция уже добавлена, не делаем этого еще раз
        if s in collected_stations:
            continue

        collected_stations.add(s)
        result_stations.append(s)

    return result_stations


def get_code_getter(request, points_to_collect):
    api_context = request.external_api_context

    code_getter = points.CodeGetter(api_context.show_systems)

    map(code_getter.collect_point, points_to_collect)

    code_getter.fetch_codes()

    return code_getter


def get_currency_info(tld, currency=None):
    geo_id = settings.TLD_TO_GEO_ID[tld]
    currencies = list(Currency.objects.all())
    src, rates = Currency.fetch_rates(currencies, geo_id)

    # Оставляем только валюты с курсом и названием
    available = {
        currency.code
        for currency in currencies
        if (currency.L_name() and currency.L_name_in() and currency.code in rates)
    }

    set_preferred = False
    if currency in available:
        selected = currency
        set_preferred = True
    else:
        selected = settings.DOMAIN_CURRENCY.get(tld)

    # Если все обломалось, используем рубли
    if not selected or selected not in available:
        selected = Currency.BASE_CURRENCY

    country_base = CurrencyInfo.country_base_from_tld(tld)
    return CurrencyInfo(
        selected,
        set_preferred,
        country_base,
        src,
        rates,
        available,
        currencies
    )
