#  -*- coding: utf-8 -*-

import operator

from django.utils import translation

from common.models.geo import Station
from travel.rasp.library.python.common23.date import environment
from common.models_utils.i18n import RouteLTitle

from travel.rasp.api_public.api_public.old_versions.core.helpers import get_code_getter


def point2json(point, code_getter=None, add_coords=False):
    if not point:
        return

    json_point = {
        'type': point.type,
        'title': point.L_title(),
        'short_title': point.L_short_title(fallback=False),
        'popular_title': point.L_popular_title(fallback=False),
        'code': point.point_key
    }

    if add_coords:
        json_point['lat'] = point.latitude
        json_point['lng'] = point.longitude

    if code_getter:
        codes = code_getter.get_codes(point)
        if codes:
            json_point['codes'] = codes

    if isinstance(point, Station):
        json_point['station_type'] = point.station_type.L_title()
        json_point['transport_type'] = point.t_type.code

    return json_point


def segments2json(segments, query, transport_subtype=False):
    request = environment.get_request()
    threads = map(operator.attrgetter('thread'), segments)

    RouteLTitle.fetch([t.L_title for t in threads])

    stations = []
    for s in segments:
        stations.append(s.station_from)
        stations.append(s.station_to)

    code_getter = get_code_getter(request, stations)
    json_segments = []
    for segment in segments:
        json_segment = Segment2Json()(segment, segment.station_from, segment.station_to, query.date, code_getter,
                                      transport_subtype)
        json_segments.append(json_segment)

    return json_segments


def t_model2json(t_model):
    lang = translation.get_language()
    if t_model:
        if lang in ['ru', 'uk']:
            title = t_model.title or t_model.title_en
        else:
            title = t_model.title_en or t_model.title

        return title


class Company2Json(object):
    def __call__(self, company, verbose=False):
        if company:
            json_company = {
                'code': company.id,
                'title': company.L_title(),
                'codes': {
                    'sirena': company.sirena_id,
                    'iata': company.iata,
                    'icao': company.icao
                }
            }

            if verbose:
                json_company['address'] = company.address
                json_company['url'] = company.url
                json_company['email'] = company.email
                json_company['contacts'] = company.contact_info
                json_company['phone'] = company.phone
                json_company['logo'] = company.logo.url if company.logo else None

            return json_company


class Thread2Json(object):
    def __init__(self, company2json=None):
        self.company2json = Company2Json() if company2json is None else company2json

    def __call__(self, thread, transport_subtype=False):
        result = {
            'number': thread.number,
            'title': thread.L_title(),
            'short_title': thread.L_title(short=True),
            'express_type': thread.express_type,
            'transport_type': thread.t_type.code,
            'carrier': self.company2json(thread.company),
            'uid': thread.uid,
            'vehicle': t_model2json(thread.t_model)
        }

        if transport_subtype:
            if thread.t_subtype:
                result['transport_subtype'] = {
                    'title': thread.t_subtype.L_title(),
                    'code': getattr(thread.t_subtype, 'code', None),
                    'color': getattr(thread.t_subtype.color, 'color', None)
                }
            else:
                result['transport_subtype'] = {'title': None, 'code': None, 'color': None}

        return result


def threadinfo2json(request, thread, path, naive_start_dt, next_plan=None, transport_subtype=False):
    first_rtstation = path[0]
    thread_start_dt = thread.pytz.localize(naive_start_dt)
    rts_start_dt = thread_start_dt.astimezone(first_rtstation.station.pytz)

    thread_to_rts_shift = (rts_start_dt.date() - thread_start_dt.date()).days
    days_text_dict = thread.L_days_text_dict(thread_to_rts_shift, naive_start_dt.date(), next_plan=next_plan)

    json_thread = Thread2Json()(thread, transport_subtype)
    json_thread.update({
        'start_time': rts_start_dt.time(),
        'days': days_text_dict.get('days_text', u''),
        'except_days': days_text_dict.get('except_days_text', u''),
        'stops': rtstation_path2json(request, path, naive_start_dt, thread.pytz)
    })

    return json_thread


def schedule_route2json(schedule_route, schedule, date, schedule_type, next_plan=None, transport_subtype=False):
    json_schedule = {
        'thread': Thread2Json()(schedule_route.thread, transport_subtype),
        'stops': schedule_route.schedule and schedule_route.schedule.L_stops(),
        'is_fuzzy': schedule_route.rtstation.is_fuzzy,
        'platform': schedule_route.rtstation.L_platform(),
        'terminal': schedule_route.rtstation.terminal_id and schedule_route.rtstation.terminal.name
    }

    days_text_dict = schedule_route.thread.L_days_text_dict(schedule_route.mask_shift,
                                                            schedule_route.naive_start_dt.date(),
                                                            next_plan=next_plan)

    json_schedule['days'] = days_text_dict.get('days_text')
    json_schedule['except_days'] = days_text_dict.get('except_days_text')

    if not date:
        json_schedule['departure_time'] = schedule_route.loc_departure_t
        json_schedule['arrival_time'] = schedule_route.loc_arrival_t

    else:
        json_schedule['departure'] = schedule_route.loc_departure_datetime
        json_schedule['arrival'] = schedule_route.loc_arrival_datetime

    if schedule_type == 'suburban':
        json_schedule['direction'] = schedule.get_direction_title_by_code(
            schedule_route.schedule_item.direction_code
        )

    return json_schedule


class Segment2Json(object):
    def __init__(self, thread2json=None):
        self.thread2json = Thread2Json() if thread2json is None else thread2json

    def __call__(self, segment, point_from, point_to, date, code_getter, transport_subtype=False):
        json_segment = {
            'thread': self.thread2json(segment.thread, transport_subtype),
            'stops': segment.L_stops(),
            'from': point2json(point_from, code_getter),
            'to': point2json(point_to, code_getter),
            'departure_platform': segment.rtstation_from.L_platform(),
            'arrival_platform': segment.rtstation_to.L_platform(),
            'departure_terminal': segment.rtstation_from.terminal_id and segment.rtstation_from.terminal.name,
            'arrival_terminal': segment.rtstation_to.terminal_id and segment.rtstation_to.terminal.name,
            'duration': segment.duration.total_seconds()
        }

        if not date:
            thread = segment.thread
            thread_tz_now = thread.pytz.localize(environment.now())

            days_text_dict = thread.L_days_text_dict(
                segment.mask_shift,
                thread_tz_now.date(),
                next_plan=getattr(segment, 'next_plan', None))

            json_segment['days'] = days_text_dict.get('days_text', u'')
            json_segment['except_days'] = days_text_dict.get('except_days_text', u'')

            json_segment['departure'] = segment.departure.strftime('%H:%M:%S')
            json_segment['arrival'] = segment.arrival.strftime('%H:%M:%S')

        else:
            json_segment['departure'] = segment.departure.strftime('%Y-%m-%d %H:%M:%S')
            json_segment['arrival'] = segment.arrival.strftime('%Y-%m-%d %H:%M:%S')

        return json_segment


def rtstation_path2json(request, path, naive_start_dt, thread_tz):
    stations_to_collect = map(operator.attrgetter('station'), path)

    code_getter = get_code_getter(request, stations_to_collect)

    start_dt = thread_tz.localize(naive_start_dt)

    rtstations = []

    for rts in path:
        departure_dt = None
        if rts.tz_departure is not None:
            departure_dt = rts.get_departure_loc_dt(naive_start_dt)

        arrival_dt = None
        if rts.tz_arrival is not None:
            arrival_dt = rts.get_arrival_loc_dt(naive_start_dt)

        stop_time = None
        if arrival_dt is not None and departure_dt is not None:
            stop_time = int((departure_dt - arrival_dt).total_seconds())

        duration = 0
        if departure_dt:
            duration = (departure_dt - start_dt).total_seconds()
        elif arrival_dt:
            duration = (arrival_dt - start_dt).total_seconds()

        rtstations.append({
            'station': point2json(rts.station, code_getter),
            'departure': departure_dt,
            'arrival': arrival_dt,
            'duration': duration,
            'stop_time': stop_time,
            'platform': rts.L_platform(),
            'terminal': rts.terminal_id and rts.terminal.name
        })

    return rtstations
