# -*- coding: utf-8 -*-

from django.db.models import Q

from common.models.geo import Station
from common.xgettext.i18n import gettext
from common.utils import geo

from travel.rasp.api_public.api_public.geobase_helpers import calc_points_distance_km
from travel.rasp.api_public.api_public.old_versions.core import helpers, json_helpers
from travel.rasp.api_public.api_public.old_versions.core.api_errors import ApiError
from travel.rasp.api_public.api_public.old_versions.core.decorators import api_view


def nearest_points(request):
    # Отдаем максимум 50 результатов
    step = 0.5  # Примерно 50 км~
    interval = step

    try:
        lat = float(request.GET.get('lat', ''))
    except ValueError:
        lat = None

    if lat is None or not (-90 <= lat <= 90):
        raise ApiError(gettext(u'Параметр "lat" должен быть числом в диапазоне от -90 до +90'),
                       http_code=400)

    try:
        lng = float(request.GET.get('lng', ''))
    except ValueError:
        lng = None

    if lng is None or not (-180 <= lng <= 180):
        raise ApiError(gettext(u'Параметр "lng" должен быть числом в диапазоне от -180 до 180'),
                       http_code=400)

    try:
        distance = float(request.GET.get('distance', '50'))
    except ValueError:
        distance = None

    if distance is None:
        raise ApiError(gettext(u'Параметр "distance" должен быть числом больше 0'),
                       http_code=400)

    center_point = Station(latitude=lat, longitude=lng)

    collected_settlements = set()
    collected_stations = set()

    settlements = []
    stations = []

    while (len(settlements) + len(stations)) < 10 and interval < 3:
        new_settlements = helpers.get_settlement_within(center_point, interval, distance,
                                                        collected_settlements)

        new_stations = helpers.get_stations_within(center_point, interval, distance,
                                                   collected_stations)

        if not (new_stations or new_settlements):
            # Если уже что то найдено и не смогли ничего добавить.
            if settlements or stations:
                break
            else:
                step *= 2
                interval += step
        else:
            settlements += new_settlements
            stations += new_stations

            interval += step

    found_points = (settlements[:50] + stations[:50])

    return found_points


@api_view
def nearest_points_view(request):
    points = nearest_points(request)
    code_getter = helpers.get_code_getter(request, points)

    json_points = []

    for p in points:
        json_point = json_helpers.point2json(p, code_getter, True)

        json_point['distance'] = p.distance

        json_points.append(json_point)

    return {
        'points': json_points,
        'deprecated': gettext(u'Пользуйтесь методом nearest_stations, это метод не'
                              u' поддерживается и будет удален.')
    }


def nearest_stations(request):
    """
    Неправильно работает, если longitude близка к 0 или 360
    """

    try:
        lat = float(request.GET.get('lat', ''))
    except ValueError:
        lat = None

    if lat is None or not (-90 <= lat <= 90):
        raise ApiError(gettext(u'Параметр "lat" должен быть числом в диапазоне от -90 до +90'),
                       http_code=400)

    try:
        lng = float(request.GET.get('lng', ''))
    except ValueError:
        lng = None

    if lng is None or not (-180 <= lng <= 180):
        raise ApiError(gettext(u'Параметр "lng" должен быть числом в диапазоне от -180 до 180'),
                       http_code=400)

    try:
        distance = float(request.GET.get('distance', '10'))
    except ValueError:
        distance = None

    if distance is None or not (0 <= distance <= 50):
        raise ApiError(gettext(u'Параметр "distance" должен быть числом от 0 до 50, по умолчанию 10'),
                       http_code=400)

    station_types = helpers.get_station_types(request)
    transport_types = helpers.get_transport_types(request)

    center_point = Station(latitude=lat, longitude=lng)

    # Берем квадрат такой чтобы вписать туда круг с радиусом distance
    (lat_top, lon_left), (lat_bottom, lon_right) = \
        geo.get_square_around_point(center_point, distance * 2)

    coords_filter = (
        Q(latitude__isnull=False, longitude__isnull=False) &
        Q(latitude__gte=lat_top, latitude__lte=lat_bottom) &
        Q(longitude__gte=lon_left, longitude__lte=lon_right)
    )

    station_qs = Station.hidden_manager.filter(coords_filter).order_by()
    if transport_types:
        station_qs = station_qs.filter(t_type__in=transport_types)

    if station_types:
        station_qs = station_qs.filter(station_type__in=station_types)

    stations = list(station_qs)

    for s in stations:
        s.distance = calc_points_distance_km(center_point, s)

    stations.sort(key=lambda _s: (_s.distance, _s.majority_id))

    return [s for s in stations if s.distance <= distance]


class NearestStationsView(helpers.ApiPaginateView):

    def handle(self, *args, **kwargs):
        stations = nearest_stations(self.request)

        pagination_json, stations = self.paginate(stations)

        code_getter = helpers.get_code_getter(self.request, stations)

        json_stations = []

        for p in stations:
            json_station = json_helpers.point2json(p, code_getter, True)

            json_station['distance'] = p.distance

            json_stations.append(json_station)

        return {
            'pagination': pagination_json,
            'stations': json_stations
        }
