# -*- coding: utf-8 -*-
import math

from travel.rasp.library.python.common23.date import environment
from common.utils.date import MSK_TZ
from common.xgettext.i18n import gettext
from stationschedule import station_schedule

from .. import helpers
from .. import points
from .. import json_helpers
from ..api_errors import ApiError


class Scheduler(object):

    LIMIT = 100

    def __init__(self, request):
        self.schedule_type = 'schedule'

        self.request = request

        self.schedule = None
        self.station = None
        self.page = None
        self.offset = None
        self.event = None
        self.schedule_date = None

    def get_schedule(self):
        try:
            self.page = int(self.request.GET.get('page', '1'))
        except ValueError:
            raise ApiError(gettext(u'Неверный номер страницы'), http_code=404)

        self.offset = (self.page - 1) * self.LIMIT
        if self.offset < 0:
            raise ApiError(gettext(u'Неверный номер страницы'), http_code=404)

        self.event = self.request.GET.get('event', u'').strip()
        if self.event not in ('arrival', 'departure', u''):
            raise ApiError(gettext(u'event должен принимать значения arrival, departure или быть пустым.'),
                           http_code=400)

        self.schedule_date = helpers.get_date_from_request(self.request, default=None)
        code_system = self.request.GET.get('system', u'yandex')
        station_code = self.request.GET.get('station', u'').strip()
        station = points.find_point(code_system, station_code, station_only=True)
        self.station = station

        t_types = helpers.get_transport_types(self.request)
        t_type_codes = None
        if t_types:
            t_type_codes = [t.code for t in t_types]

        # Если это расписание электричек
        kwargs = {}
        if t_type_codes and len(t_type_codes) == 1 and t_type_codes[0] == 'suburban':
            self.schedule_type = 'suburban'
            kwargs = dict(requested_direction=self.request.GET.get('direction'),
                          default_direction='all')
        else:
            if not self.event:
                self.event = 'departure'

        self.schedule = station_schedule(
            station, self.schedule_type, t_type_code=t_type_codes,
            event=self.event, limit=self.LIMIT, offset=self.offset,
            schedule_date=self.schedule_date,
            **kwargs)

        return self.schedule


class ScheduleView(helpers.ApiPaginateView):

    def handle(self, *args, **kwargs):
        scheduler = Scheduler(self.request)
        schedule = scheduler.get_schedule()

        msk_today = environment.now_aware().astimezone(MSK_TZ).date()
        current_plan, next_plan = schedule.current_next_plans(msk_today)

        station = scheduler.station
        code_getter = helpers.get_code_getter(self.request, [station])

        transport_subtype = self.request.GET.get('transport_subtype', False) == 'true'
        if not transport_subtype:
            transport_subtype = kwargs.pop('transport_subtype', False)

        pagination_json = {
            'total': schedule.total,
            'page': scheduler.page,
            'has_next': bool((scheduler.offset + scheduler.LIMIT) < schedule.total),
            'per_page': scheduler.LIMIT,
            'page_count': int(math.ceil(schedule.total / float(scheduler.LIMIT)))
        }

        result = {
            'event': scheduler.event,
            'date': scheduler.schedule_date,
            'pagination': pagination_json,
            'station': json_helpers.point2json(station, code_getter),
            'schedule': [
                json_helpers.schedule_route2json(sr, schedule, scheduler.schedule_date, scheduler.schedule_type,
                                                 next_plan, transport_subtype)
                for sr in schedule.schedule
            ]
        }

        if scheduler.schedule_type == 'suburban':
            del result['event']
            result['directions'] = schedule.direction_title_by_code

            # Хак для xml
            # TODO: в следующей вресии сделать тоже самое для json

            out_format = self.request.GET.get('format', 'json')

            if out_format == 'xml':
                result['directions'] = [
                    {'code': c, 'title': t} for c, t in schedule.direction_title_by_code.items()
                ]

            result['direction'] = {
                'code': schedule.direction_code,
                'title': schedule.direction_title
            }

        return result
