# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import re
import json
import logging

from django.utils.html import escape
from django.utils.encoding import smart_str
from rest_framework import status
from rest_framework.response import Response

from travel.rasp.api_public.api_public.v3.core.helpers import serialize_date_time


log = logging.getLogger(__name__)

VALID_CALLBACK_RE = re.compile(r'^[$A-Za-z\d_]*$')


def get_jsonp_response(result, http_code):
    if status.HTTP_200_OK == http_code:
        return {'status': 'success',
                'data': result}
    else:
        return {'status': 'error',
                'message': result['error']['text'],
                'data': {'error_code': result['error']['error_code'],
                         'request': result['error']['request']},
                'code': http_code}


def build_response(request, result, http_code):
    out_format = request.GET.get('format', 'json')
    callback = request.GET.get('callback', u'').strip()

    if isinstance(result, basestring):
        response = Response(result, status=http_code)
        response['X-Content-Type-Options'] = 'nosniff'
        return response

    if out_format == 'xml':
        response = Response(result, status=http_code)
        response['X-Content-Type-Options'] = 'nosniff'
        return response

    if out_format == 'json':
        return jsonp_wrap(result, callback, http_code)
    else:
        response = Response({'status': 'error',
                             'message': u'Указан неправильный формат ответа - {}.'.format(smart_str(escape(out_format))),
                             'code': status.HTTP_400_BAD_REQUEST})
        response['X-Content-Type-Options'] = 'nosniff'
        return response


def jsonp_wrap(result, callback, http_code):
    if not callback:
        json_str = json.dumps(result, ensure_ascii=False, default=serialize)
        result = json.loads(json_str)
        response = Response(result, status=http_code, content_type='application/json; charset=utf-8')
        response['X-Content-Type-Options'] = 'nosniff'
        return response

    if not VALID_CALLBACK_RE.match(callback):
        result = {'status': 'error',
                  'message': u'Указано неправильное имя для callback - {}.'.format(smart_str(callback)),
                  'code': status.HTTP_400_BAD_REQUEST}
        json_str = json.dumps(result, ensure_ascii=False, default=serialize)
        js_string = '{}({});'.format(smart_str(callback), smart_str(json_str))
        response = Response(js_string, content_type='text/plain; charset=utf-8')
        response['X-Content-Type-Options'] = 'nosniff'
        return response

    # Для json_p всегда посылаем 200
    result = get_jsonp_response(result, http_code)
    json_str = json.dumps(result, ensure_ascii=False, default=serialize)
    js_string = '{}({});'.format(smart_str(callback), smart_str(json_str))
    response = Response(js_string, status=status.HTTP_200_OK, content_type='text/plain; charset=utf-8')
    response['X-Content-Type-Options'] = 'nosniff'
    return response


def serialize(obj):
    if hasattr(obj, '__json__'):
        return obj.__json__()

    datetime_obj = serialize_date_time(obj)
    if datetime_obj:
        return datetime_obj

    raise TypeError(repr(obj) + " is not JSON serializable")
