# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.core.exceptions import ObjectDoesNotExist

from common.models.geo import Point, Station, Settlement, StationCode
from common.xgettext.i18n import gettext

from .api_errors import ApiError, ApiInternalError


ALLOWED_SYSTEMS = ('yandex', 'iata', 'sirena', 'express', 'esr')


class PointFinder(object):
    @classmethod
    def find_point(cls, system, code, station_only=False):
        if system not in ALLOWED_SYSTEMS:
            raise ApiError(gettext(u'Неизвестная систем кодирования {code_system}')
                           .format(code_system=system), http_code=400)

        point = getattr(cls, 'find_by_{}'.format(system))(code, station_only)
        if not point:
            if station_only:
                raise ApiError(gettext(u'Не нашли станцию по {system} коду {code}.').format(
                    code=code, system=system), http_code=404)
            else:
                raise ApiError(gettext(u'Не нашли объект по {system} коду {code}').format(
                    code=code, system=system), http_code=404)

        return point

    @classmethod
    def find_by_yandex(cls, code, station_only):
        if len(code) < 1:
            return

        try:
            point = Point.get_by_key(code)
        except (ValueError, ObjectDoesNotExist):
            return

        if not isinstance(point, (Station, Settlement)):
            return

        if station_only:
            if isinstance(point, Station):
                return point
            else:
                return

        return point

    @classmethod
    def find_by_iata(cls, code, station_only):
        if not station_only:
            try:
                return Settlement.objects.get(iata=code)
            except Settlement.DoesNotExist:
                pass

        try:
            return Station.get_by_code('iata', code)
        except Station.DoesNotExist:
            pass

    @classmethod
    def find_by_sirena(cls, code, station_only):
        if not station_only:
            try:
                return Settlement.objects.get(sirena_id=code)
            except Settlement.DoesNotExist:
                pass

        try:
            return Station.get_by_code('sirena', code)
        except Station.DoesNotExist:
            pass

    @classmethod
    def find_by_express(cls, code, station_only):
        try:
            station = Station.get_by_code('express', code)
            if station.majority.code != 'express_fake':
                return station

            # Это псевдостанция. Она соответствует городу.
            if station_only:
                return

            return station.settlement

        except (Station.DoesNotExist, Settlement.DoesNotExist):
            pass

    @classmethod
    def find_by_esr(cls, code, station_only):
        try:
            return Station.get_by_code('esr', code)
        except Station.DoesNotExist:
            pass


find_point = PointFinder.find_point


class CodeGetter(object):
    def __init__(self, systems=ALLOWED_SYSTEMS):
        self.systems = systems or []
        self._stations = set()
        self._settlements = set()
        self._codes_by_yandex = {}

    def collect_point(self, point):
        if not self.systems:
            return

        if isinstance(point, Station):
            self._stations.add(point)
        elif isinstance(point, Settlement):
            self._settlements.add(point)
        else:
            raise ApiInternalError(u'Отдаем наружу только станции и города')

    def get_codes(self, point):
        if not self.systems:
            return

        return self._codes_by_yandex[point.point_key]

    def fetch_codes(self):
        if not self.systems:
            return

        self._pre_fill()

        for system in self.systems:
            getattr(self, '_fetch_{}'.format(system))()

    def _pre_fill(self):
        for s in self._stations:
            self._codes_by_yandex[s.point_key] = {}

        for s in self._settlements:
            self._codes_by_yandex[s.point_key] = {}

    def _fetch_yandex(self):
        for ya_code, codes in self._codes_by_yandex.iteritems():
            codes['yandex'] = ya_code

    def _fetch_iata(self):
        for station_id, iata in StationCode.objects \
                .filter(system__code='iata', station__in=self._stations) \
                .values_list('station_id', 'code'):
            ya_code = Point.get_point_key(Station, station_id)

            self._codes_by_yandex[ya_code]['iata'] = iata

        for settlement in self._settlements:
            self._codes_by_yandex[settlement.point_key]['iata'] = settlement.iata

    def _fetch_sirena(self):
        for station_id, sirena in StationCode.objects \
                .filter(system__code='sirena', station__in=self._stations) \
                .values_list('station_id', 'code'):
            ya_code = Point.get_point_key(Station, station_id)

            self._codes_by_yandex[ya_code]['sirena'] = sirena

        for settlement in self._settlements:
            self._codes_by_yandex[settlement.point_key]['sirena'] = settlement.sirena_id

    def _fetch_express(self):
        for station_id, express in StationCode.objects \
                .filter(system__code='express', station__in=self._stations) \
                .values_list('station_id', 'code'):
            ya_code = Point.get_point_key(Station, station_id)

            self._codes_by_yandex[ya_code]['express'] = express

        for station_id, express, settlement_id in \
                StationCode.objects.filter(station__settlement__in=self._settlements) \
                        .filter(system__code='express', station__majority__code='express_fake') \
                        .values_list('station_id', 'code', 'station__settlement_id'):
            ya_code = Point.get_point_key(Settlement, settlement_id)

            self._codes_by_yandex[ya_code]['express'] = express

    def _fetch_esr(self):
        for station_id, esr in StationCode.objects \
                .filter(system__code='esr', station__in=self._stations) \
                .values_list('station_id', 'code'):
            ya_code = Point.get_point_key(Station, station_id)

            self._codes_by_yandex[ya_code]['esr'] = esr
