# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.db.models import Q

from common.models.geo import Settlement
from common.utils.geo import GeoPoint
from common.xgettext.i18n import gettext

from travel.rasp.api_public.api_public.v3.core import api_errors
from travel.rasp.api_public.api_public.v3.nearest.external_urls import (
    get_station_view_url_desktop, get_station_view_url_touch
)
from travel.rasp.api_public.api_public.v3.core.helpers import get_code_getter
from travel.rasp.api_public.api_public.v3.nearest.data_layer import nearest_stations
from travel.rasp.api_public.api_public.v3.views.base_api_view import BaseApiView
from travel.rasp.api_public.api_public.v3.views.json_helpers import point2json
from travel.rasp.api_public.api_public.v3.views.paginate_view import ApiPaginateView
from travel.rasp.api_public.api_public.v3.request_schemas import NearestPointRequestSchema


class NearestStationsView(ApiPaginateView):
    def handle(self, request, *args, **kwargs):
        query, errors = NearestPointRequestSchema(context={'request': request}).load(request.GET)
        stations = nearest_stations(query['lat'], query['lng'], query['distance'],
                                    query['station_types'], query['transport_types'])

        pagination_json, stations = self.paginate(stations, request)

        code_getter = get_code_getter(stations, query['show_systems'])
        json_stations = []
        for station in stations:
            json_station = point2json(station, code_getter, True)

            json_station['distance'] = station.distance

            # RASPAPI-522
            json_station['majority'] = station.majority_id
            json_station['type_choices'] = {
                type_choice: {
                    'desktop_url': get_station_view_url_desktop(station.id, type_choice),
                    'touch_url': get_station_view_url_touch(station.id, type_choice),
                }
                for type_choice in station.type_choices_set
                }

            json_stations.append(json_station)

        return {
            'pagination': pagination_json,
            'stations': json_stations
        }


class NearestSettlement(BaseApiView):
    def handle(self, request, *args, **kwargs):
        query, errors = NearestPointRequestSchema(context={'request': request}).load(request.GET)
        center_point = GeoPoint(latitude=query['lat'], longitude=query['lng'])

        # Берем квадрат такой чтобы вписать туда круг с радиусом distance
        top_left_point, bottom_right_point = center_point.expand_square(query['distance'])

        settlements = Settlement.hidden_manager.filter(
            Q(latitude__isnull=False, longitude__isnull=False) &
            Q(latitude__gte=top_left_point.latitude,
              latitude__lte=bottom_right_point.latitude) &
            Q(longitude__gte=top_left_point.longitude,
              longitude__lte=bottom_right_point.longitude)
        ).order_by()

        if not settlements:
            raise api_errors.ApiError(gettext(u'Населенный пункт не найден'),
                                      http_code=404)

        settlement_distance, _majority_id, settlement = min(
            (
                center_point.distance(settlement),
                settlement.majority_id,
                settlement
            )
            for settlement in settlements
        )

        if settlement_distance > query['distance']:
            raise api_errors.ApiError(gettext(u'Населенный пункт не найден'),
                                      http_code=404)

        return dict(
            point2json(settlement, add_coords=True),
            distance=settlement_distance
        )
