# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from datetime import timedelta, datetime, time

from django.utils import translation

from common.data_api.baris.instance import baris
from common.data_api.baris.helpers import get_plane_stations_ids, BarisData

from travel.rasp.api_public.api_public.v3.search.baris_segments import OneDayBarisSegment, AllDaysBarisSegment
from travel.rasp.api_public.api_public.v3.search.base_search import BaseSearch


class BaseBarisSearch(BaseSearch):
    """Базовый поиск в БАРиС"""
    def __init__(self, query, request, query_points):
        super(BaseBarisSearch, self).__init__(request, query, query_points)

        self.station_from_ids = get_plane_stations_ids(self.point_from)
        self.station_to_ids = get_plane_stations_ids(self.point_to)

    def make_used_points(self, baris_data):
        self.used_points = set(baris_data.stations_by_ids.values())
        self.used_points.update(baris_data.settlements_by_ids.values())


class OneDayBarisSearch(BaseBarisSearch):
    """Поиск в БАРиС на один день"""
    def __init__(self, query, request, query_points):
        super(OneDayBarisSearch, self).__init__(query, request, query_points)

    def search(self):
        after = datetime.combine(self.departure_date, time(0))
        before = datetime.combine(self.departure_date + timedelta(days=1, hours=4), time(0))
        baris_data = BarisData(baris.get_p2p_search(
            self.station_from_ids, self.station_to_ids,
            after=after, before=before, national_version=self.national_version
        ))
        self.make_used_points(baris_data)

        self.segments = [
            OneDayBarisSegment(flight, baris_data)
            for flight in baris_data.flights
        ]
        self._add_flights_run_days(self.segments)

    def _add_flights_run_days(self, segments):
        """Добавление календаря дней хождения в рейсы поиска на день"""
        if not self.add_days_mask or not segments:
            return None

        # Для получения дней хождения приходится дергать ручку поиска в БАРиС на все дни
        result_timezone = self.result_pytz.zone if self.result_pytz else None
        all_days_response = baris.get_p2p_all_days_search(
            translation.get_language(), self.station_from_ids, self.station_to_ids,
            self.point_from.time_zone, self.point_to.time_zone, self.national_version,
            result_timezone=result_timezone, add_run_days=True, add_days_text=False
        )

        segments_by_numbers = {s.number: s for s in segments}

        for all_days_flight in all_days_response.flights:
            if all_days_flight["title"] in segments_by_numbers:
                segment = segments_by_numbers[all_days_flight["title"]]
                departure = segment.departure.date()

                if str(departure.year) in all_days_flight["runDays"]:
                    year_months = all_days_flight["runDays"][str(departure.year)]

                    if str(departure.month) in year_months:
                        month_days = year_months[str(departure.month)]

                        if month_days[departure.day - 1]:
                            segment.set_all_days_flight(all_days_flight)


class AllDaysBarisSearch(BaseBarisSearch):
    """Поиск в БАРиС на все дни"""
    def __init__(self, query, request, query_points):
        super(AllDaysBarisSearch, self).__init__(query, request, query_points)

    def search(self):
        result_timezone = self.result_pytz.zone if self.result_pytz else None
        baris_data = BarisData(baris.get_p2p_all_days_search(
            translation.get_language(), self.station_from_ids, self.station_to_ids,
            self.point_from.time_zone, self.point_to.time_zone, self.national_version,
            result_timezone=result_timezone, add_run_days=self.add_days_mask
        ))
        self.make_used_points(baris_data)

        self.segments = [
            AllDaysBarisSegment(flight, baris_data)
            for flight in baris_data.flights
        ]
