# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from dateutil import parser

from common.data_api.baris.base_avia_segment import BaseAviaSegment
from common.data_api.baris.helpers import make_pseudo_uid_for_baris_flight
from common.utils.title_generator import DASH
from route_search.transfers.transfer_segment import BarisTransferSegment

from travel.rasp.api_public.api_public.v3.views.json_helpers import Company2Json, t_model2json, point2json
from travel.rasp.api_public.api_public.v3.search.helpers import make_thread_method_link, make_segment_mask_schedule


class OneDayBarisSegment(BaseAviaSegment):
    """Сегмент авиарейса с данными на конкретный день"""
    def __init__(self, baris_flight, baris_data):
        super(OneDayBarisSegment, self).__init__(baris_flight, baris_data, title_separator=DASH)

        self.departure = parser.parse(baris_flight["departureDatetime"])
        self.arrival = parser.parse(baris_flight["arrivalDatetime"])
        if "startDatetime" in baris_flight and baris_flight["startDatetime"]:
            self.start = parser.parse(baris_flight["startDatetime"])
        else:
            self.start = self.departure

    def set_all_days_flight(self, all_days_flight):
        self.run_days = all_days_flight["tzRunDays" if "tzRunDays" in all_days_flight else "runDays"]


class AllDaysBarisSegment(BaseAviaSegment):
    """Сегмент авиарейса с данными на все дни"""
    def __init__(self, baris_flight, baris_data):
        super(AllDaysBarisSegment, self).__init__(baris_flight, baris_data, title_separator=DASH)

        if "tzDaysText" in baris_flight:
            self.days_text = baris_flight["tzDaysText"]
        else:
            self.days_text = baris_flight["daysText"]

        if "tzRunDays" in baris_flight:
            self.run_days = baris_flight["tzRunDays"]
        elif "runDays" in baris_flight:
            self.run_days = baris_flight["runDays"]

        self.departure = baris_flight["departure"]
        self.arrival = baris_flight["arrival"]

        start_pytz = baris_data.stations_by_ids[baris_flight["route"][0]].pytz
        self.start = start_pytz.localize(baris_flight["naiveStart"])


def _get_tz_times(segment, result_pytz):
    """Перевод временных параметров сегмента в указанную таймзону. Для пересадок не производится"""
    if not isinstance(segment, BarisTransferSegment) and result_pytz:
        segment.departure = segment.departure.astimezone(result_pytz)
        segment.arrival = segment.arrival.astimezone(result_pytz)
        segment.start = segment.start.astimezone(result_pytz)


def baris_segment2json(segment, code_getter, thread_url=None, result_pytz=None, point_from=None, point_to=None):
    """Формирование json по авиасегменту"""
    start_date = segment.start.date()
    uid = make_pseudo_uid_for_baris_flight(
        segment.number, segment.company.id if hasattr(segment, "company") else 0, start_date
    )
    is_transfer = isinstance(segment, BarisTransferSegment)

    _get_tz_times(segment, result_pytz)

    if not point_from:
        point_from = segment.station_from
    if not point_to:
        point_to = segment.station_to

    json_segment = {
        "thread": {
            "number": segment.number,
            "title": segment.title,
            "short_title": segment.title,
            "carrier": Company2Json(
                verbose=not is_transfer, logo_svg=not is_transfer
            )(getattr(segment, "company", None)),
            "vehicle": t_model2json(getattr(segment.transport, "model", None)),
            "express_type": None,
            "transport_type": "plane",
            "transport_subtype": {"title": None, "code": None, "color": None},
            "uid": uid,
        },
        "from": point2json(point_from, code_getter),
        "to": point2json(point_to, code_getter),
        "departure_platform": "",
        "arrival_platform": "",
        "departure_terminal": segment.departure_terminal,
        "arrival_terminal": segment.arrival_terminal,
        "stops": "",
        "duration": (segment.arrival - segment.departure).total_seconds(),
        "start_date": segment.start.date().strftime("%Y-%m-%d")
    }

    if thread_url:
        json_segment["thread"]["thread_method_link"] = make_thread_method_link(thread_url, start_date, uid)

    if hasattr(segment, "run_days"):
        json_segment["schedule"] = make_segment_mask_schedule(segment.run_days)

    if isinstance(segment, AllDaysBarisSegment):
        json_segment.update({
            "departure": segment.departure.strftime("%H:%M:%S"),
            "arrival": segment.arrival.strftime("%H:%M:%S"),
            "days": segment.days_text,
            "except_days": "",
        })
    elif isinstance(segment, OneDayBarisSegment):
        json_segment.update({
            "departure": segment.departure.isoformat(),
            "arrival": segment.arrival.isoformat(),
            "has_transfers": False,
            "tickets_info": {"et_marker": False, "places": []},
        })
    elif is_transfer:  # Сегмент из варианта с пересадками
        json_segment.update({
            "departure": segment.departure.isoformat(),
            "arrival": segment.arrival.isoformat()
        })

    return json_segment
