# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from datetime import datetime
from urllib import urlencode

from common.models.schedule import RThread, RTStation
from route_search.transfers.variant import Variant
from route_search.models import RThreadSegment


def make_segment_mask_schedule(run_days):
    """Преобразует календарь дней хождения в формат вывода api_public"""
    segment_schedule = []
    for year, months in run_days.items():
        for month, days in months.items():
            segment_schedule.append({"year": year, "month": month, "days": days})
    segment_schedule.sort(key=lambda x: int(x["month"]))
    return segment_schedule


def make_thread_method_link(thread_link_url, start_date, uid):
    """Формирует ссылку на ручку нитки"""
    params = urlencode({
        "date": start_date,
        "uid": uid
    })
    return "".join((thread_link_url, params))


def _add_local_start_dt_to_segment(segment, first_rtstations_by_threads):
    thread = segment.thread
    if isinstance(thread, RThread):
        first_rtstation = first_rtstations_by_threads[thread]
        naive_start_dt = datetime.combine(segment.start_date, thread.tz_start_time)
        segment.thread_local_start_dt = first_rtstation.get_departure_loc_dt(naive_start_dt)


def fill_thread_local_start_dt(segments):
    """Добавляет локальное время отправления с исходной станции для сегментов из базы расписаний"""
    threads = set()
    for segment in segments:
        if isinstance(segment, Variant):
            threads.update(s.thread for s in segment.segments if isinstance(s.thread, RThread))
        elif isinstance(segment, RThreadSegment):
            threads.add(segment.thread)

    first_rtstations = (
        RTStation.objects.filter(tz_arrival__isnull=True, thread__in=list(threads)).select_related("station", "thread")
    )

    first_rtstations_by_threads = {rts.thread: rts for rts in first_rtstations}
    for segment in segments:
        if isinstance(segment, Variant):
            for variant_segment in segment.segments:
                _add_local_start_dt_to_segment(variant_segment, first_rtstations_by_threads)

        elif isinstance(segment, RThreadSegment):
            _add_local_start_dt_to_segment(segment, first_rtstations_by_threads)
