# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import datetime

from travel.rasp.library.python.common23.date import environment
from common.views.thread import get_interval_string

from travel.rasp.api_public.api_public.v3.search.helpers import make_thread_method_link
from travel.rasp.api_public.api_public.v3.views.json_helpers import (
    Thread2Json, point2json, Company2Json, tickets_info2json
)


def common_segment2json(thread2json, segment, point_from, point_to, code_getter):
    """Базовый json для всех сегментов"""
    return {
        "thread": thread2json(segment.thread),
        "stops": segment.L_stops(),
        "from": point2json(point_from, code_getter),
        "to": point2json(point_to, code_getter),
        "departure_platform": segment.rtstation_from.L_platform(),
        "arrival_platform": segment.rtstation_to.L_platform(),
        "departure_terminal": segment.rtstation_from.terminal_id and segment.rtstation_from.terminal.name,
        "arrival_terminal": segment.rtstation_to.terminal_id and segment.rtstation_to.terminal.name,
        "duration": segment.duration.total_seconds()
    }


def common_rasp_db_segment2json(segment, point_from, point_to, date, code_getter, result_pytz, currency_info):
    """Базовый json для всех сегментов из базы расписаний"""
    thread2json = Thread2Json(Company2Json(verbose=True, logo_svg=True))
    json_segment = common_segment2json(thread2json, segment, point_from, point_to, code_getter)

    if getattr(segment, "schedule", None):
        json_segment["schedule"] = segment.schedule

    if not date:  # Сегмент поиска на все дни
        thread = segment.thread
        thread_tz_now = thread.pytz.localize(environment.now())

        shift = segment.rtstation_from.calc_days_shift(
            event="departure",
            start_date=segment.start_date,
            event_tz=result_pytz
        )

        days_text_dict = thread.L_days_text_dict(
            shift,
            thread_tz_now.date(),
            next_plan=getattr(segment, "next_plan", None))

        json_segment["days"] = days_text_dict.get("days_text", "")
        json_segment["except_days"] = days_text_dict.get("except_days_text", "")

    if date:  # Сегмент поиска на день
        json_segment["has_transfers"] = False
        json_segment["tickets_info"] = tickets_info2json(segment, currency_info)

    return json_segment


def rasp_db_segment2json(
    segment, point_from, point_to, date, code_getter, thread_url, result_pytz=None, currency_info=None
):
    """json для не интервального сегмента базы расписаний"""
    json_segment = common_rasp_db_segment2json(
        segment, point_from, point_to, date, code_getter, result_pytz, currency_info
    )

    seg_departure = segment.departure.astimezone(result_pytz) if result_pytz else segment.departure
    seg_arrival = segment.arrival.astimezone(result_pytz) if result_pytz else segment.arrival

    if not date:  # Сегмент поиска на все дни
        json_segment["departure"] = seg_departure.strftime("%H:%M:%S")
        json_segment["arrival"] = seg_arrival.strftime("%H:%M:%S")

    else:  # Сегмент поиска на день
        json_segment["departure"] = seg_departure.isoformat()
        json_segment["arrival"] = seg_arrival.isoformat()

    json_segment["start_date"] = (
        segment.thread_local_start_dt.astimezone(result_pytz)
        if result_pytz
        else segment.thread_local_start_dt
    ).strftime("%Y-%m-%d")

    json_segment["thread"]["thread_method_link"] = make_thread_method_link(
        thread_url, segment.thread_local_start_dt.date(), json_segment["thread"]["uid"]
    )

    return json_segment


def interval_segment2json(
    segment, point_from, point_to, date, code_getter, thread_url, result_pytz=None, currency_info=None
):
    """json для интервального сегмента"""
    json_segment = common_rasp_db_segment2json(
        segment, point_from, point_to, date, code_getter, result_pytz, currency_info
    )

    json_segment["thread"]["interval"] = {"density": get_interval_string(segment.thread)}

    if not date:  # Сегмент поиска на все дни
        json_segment["thread"]["interval"].update({
            "begin_time": segment.thread.begin_time,
            "end_time": segment.thread.end_time
        })
        json_segment["start_date"] = segment.thread.first_run(
            environment.now_aware().astimezone(segment.station_from.pytz).date()
        ).strftime("%Y-%m-%d")

    else:  # Сегмент поиска на день
        json_segment["thread"]["interval"].update({
            "begin_time": datetime.combine(date, segment.thread.begin_time).isoformat(),
            "end_time": datetime.combine(date, segment.thread.end_time).isoformat()
        })
        json_segment["start_date"] = date.strftime("%Y-%m-%d")

    json_segment["thread"]["thread_method_link"] = make_thread_method_link(
        thread_url, json_segment["start_date"], json_segment["thread"]["uid"]
    )

    return json_segment
