# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import operator

from common.utils.title_generator import DASH
from route_search.transfers.transfer_segment import BarisTransferSegment
from route_search.transfers.transfers import get_transfer_variants

from travel.rasp.api_public.api_public.v3.search.base_search import BaseSearch
from travel.rasp.api_public.api_public.v3.search.baris_segments import baris_segment2json
from travel.rasp.api_public.api_public.v3.views.json_helpers import point2json, Thread2Json, Company2Json
from travel.rasp.api_public.api_public.v3.search.rasp_db_json_helpers import common_segment2json


class TransfersSearch(BaseSearch):
    """Поиск вариантов пересадок"""
    def __init__(self, query, request, query_points):
        super(TransfersSearch, self).__init__(request, query, query_points)

        self.t_types = query["transport_types"]

    def search(self):
        transfer_tcodes = [t.code for t in self.t_types] if self.t_types else None
        self.segments = list(get_transfer_variants(
            self.point_from, self.point_to, self.departure_date,
            transfer_tcodes, baris_title_separator=DASH
        ))

        for variant in self.segments:
            for segment in variant.segments:
                self.add_segment_used_points(segment)


def _transfer_segment2json(segment, point_from, point_to, code_getter):
    """json для не авиа сегмента из варианта пересадок"""
    thread2json = Thread2Json(Company2Json())
    json_segment = common_segment2json(thread2json, segment, point_from, point_to, code_getter)

    json_segment.update({
        "departure": segment.departure.isoformat(),
        "arrival": segment.arrival.isoformat(),
        "start_date": segment.thread_local_start_dt.strftime("%Y-%m-%d")
    })

    return json_segment


def _variant2json(variant, code_getter):
    return {
        "departure_from": point2json(variant.station_from),
        "arrival_to": point2json(variant.station_to),
        "transport_types": map(operator.attrgetter("code"), variant.transport_types),
        "departure": variant.departure.isoformat(),
        "arrival": variant.arrival.isoformat(),
        "transfers": [point2json(transfer_point, code_getter) for transfer_point in variant.transfers]
    }


def transfer_variant2json(variant, code_getter):
    transfer_segments = []

    for i, transfer_segment in enumerate(variant.segments):
        # В качестве точек сегментов пересадок, в отличии от обычных сегментов, по-возможности берутся города
        point_from = transfer_segment.station_from.settlement or transfer_segment.station_from
        point_to = transfer_segment.station_to.settlement or transfer_segment.station_to

        if point_from == point_to:
            point_from = transfer_segment.station_from
            point_to = transfer_segment.station_to

        # Сегмент варианта может быть из БАРиС или из базы расписаний
        if isinstance(transfer_segment, BarisTransferSegment):
            json_trans_segment = baris_segment2json(
                transfer_segment, code_getter, point_from=point_from, point_to=point_to
            )
        else:
            json_trans_segment = _transfer_segment2json(transfer_segment, point_from, point_to, code_getter)

        transfer_segments.append(json_trans_segment)

        # Добавляем пересадку как сегмент варианта
        transfer = transfer_segment.display_info.get("transfer")
        if transfer:
            json_transfer = {
                "is_transfer": True,
                "duration": transfer["duration"].total_seconds(),
                "transfer_point": point2json(transfer["in"], code_getter),
                "transfer_from": point2json(transfer.get("from"), code_getter),
                "transfer_to": point2json(transfer.get("to"), code_getter)
            }
            transfer_segments.append(json_transfer)

    json_variant = _variant2json(variant, code_getter)
    json_variant["has_transfers"] = True
    json_variant["details"] = transfer_segments

    return json_variant
