# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from common.models.transport import TransportType
from common.xgettext.i18n import gettext
from common.data_api.baris.base_avia_segment import BaseAviaSegment
from route_search.models import IntervalRThreadSegment
from route_search.transfers.variant import Variant

from travel.rasp.api_public.api_public.v3.search.baris_search import OneDayBarisSearch, AllDaysBarisSearch
from travel.rasp.api_public.api_public.v3.search.baris_segments import baris_segment2json
from travel.rasp.api_public.api_public.v3.search.rasp_db_search import OneDayRaspDbSearch, AllDaysRaspDbSearch
from travel.rasp.api_public.api_public.v3.search.transfers_search import TransfersSearch, transfer_variant2json
from travel.rasp.api_public.api_public.v3.core.helpers import get_code_getter, get_currency_info
from travel.rasp.api_public.api_public.v3.search.helpers import fill_thread_local_start_dt
from travel.rasp.api_public.api_public.v3.views.paginate_view import ApiPaginateView
from travel.rasp.api_public.api_public.v3.views.json_helpers import point2json
from travel.rasp.api_public.api_public.v3.request_schemas import SearchRequestSchema
from travel.rasp.api_public.api_public.v3.core import points
from travel.rasp.api_public.api_public.v3.core.api_errors import ApiError
from travel.rasp.api_public.api_public.v3.search.rasp_db_json_helpers import rasp_db_segment2json, interval_segment2json


class SearchQueryPoints(object):
    """Класс для формирования точек и даты отправления по исходным параметрам запроса"""
    def __init__(self, query):
        self.point_from = points.PointFinder.find_point(query["system_from"], query["code_from"])
        self.point_to = points.PointFinder.find_point(query["system_to"], query["code_to"])

        if self.point_to == self.point_from:
            raise ApiError(gettext("Точка отправления не должна совпадать с точкой прибытия."), http_code=400)

        self.departure_date = self.point_from.localize(loc=query["dt"]).date() if query["dt"] else None


class SearchView(ApiPaginateView):
    """
    Ручка поиска
    https://st.yandex-team.ru/RASPFRONT-8849
    """
    def handle(self, request, *args, **kwargs):
        query, errors = SearchRequestSchema(context={"request": request}).load(request.GET)

        query_points = SearchQueryPoints(query)
        search_on_day = bool(query["dt"])
        t_types = query["transport_types"]
        plane = TransportType.get_plane_type()
        need_baris = not t_types or plane in t_types
        need_rasp_db = not t_types or t_types != [plane]
        need_transfers = query["transfers"]

        used_points = {query_points.point_from, query_points.point_to}
        segments = []
        interval_segments = []
        currency_info = None

        def process_searcher(searcher):
            searcher.search()
            segments.extend(searcher.segments)
            interval_segments.extend(searcher.interval_segments)
            used_points.update(searcher.used_points)

        if search_on_day:
            if need_baris:
                process_searcher(OneDayBarisSearch(query, request, query_points))
            if need_rasp_db:
                # Цены добавляются только в сегменты расписаний на один день
                currency_info = get_currency_info(request.tld, query["currency"])
                process_searcher(OneDayRaspDbSearch(query, request, query_points, currency_info))

            # Сортируем вместе не интервальные сегменты из базы и БАРиС, и в конец списка добавляем пересадки
            # Так работало до RASPFRONT-8849, так и оставили
            segments.sort(key=lambda s: s.departure)

            if need_transfers:
                process_searcher(TransfersSearch(query, request, query_points))

        else:
            if need_baris:
                process_searcher(AllDaysBarisSearch(query, request, query_points))
            if need_rasp_db:
                process_searcher(AllDaysRaspDbSearch(query, request, query_points))
            segments.sort(key=lambda s: s.departure.time())

        fill_thread_local_start_dt(segments)

        # Интервальные сегменты идут последними по списку, при пейджинге это также учитывается
        segments.extend(interval_segments)
        pagination_json, paginated_segments = self.paginate(segments, request)

        code_getter = get_code_getter(used_points, query["show_systems"])
        thread_url = "{}/{}".format(request.host, "v3/thread/?")

        json_segments, json_interval_segments = segments2json(
            paginated_segments, query_points.departure_date, code_getter,
            thread_url, query["result_pytz"], currency_info
        )

        result = {
            "search": {
                "from": point2json(query_points.point_from, code_getter),
                "to": point2json(query_points.point_to, code_getter),
                "date": query_points.departure_date,
            },
            "segments": json_segments,
            "interval_segments": json_interval_segments,
            "pagination": pagination_json
        }

        return result


def segments2json(segments, departure_date, code_getter, thread_url, result_pytz, currency_info):
    """
    Формирование json для списка сегментов
    :return: списки простых сегментов, включая пересадки, и список интервальных сегментов
    """
    json_segments, json_interval_segments = [], []

    for segment in segments:
        if isinstance(segment, Variant):
            json_segments.append(
                transfer_variant2json(segment, code_getter)
            )

        elif isinstance(segment, BaseAviaSegment):
            json_segments.append(
                baris_segment2json(segment, code_getter, thread_url, result_pytz)
            )

        elif isinstance(segment, IntervalRThreadSegment):
            json_interval_segments.append(
                interval_segment2json(
                    segment, segment.station_from, segment.station_to, departure_date, code_getter,
                    thread_url, result_pytz, currency_info
                )
            )

        else:
            json_segments.append(
                rasp_db_segment2json(
                    segment, segment.station_from, segment.station_to, departure_date, code_getter,
                    thread_url, result_pytz, currency_info
                )
            )

    return json_segments, json_interval_segments
