# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from datetime import datetime, time, timedelta
from dateutil import parser

from django.utils import translation

from common.data_api.baris.instance import baris
from common.data_api.baris.helpers import BarisData, make_pseudo_uid_for_baris_flight, get_flight_title

from travel.rasp.api_public.api_public.v3.views.json_helpers import Company2Json, t_model2json
from travel.rasp.api_public.api_public.v3.station.rasp_db_scheduler import BaseScheduler

log = logging.getLogger(__name__)


class BaseBarisScheduler(BaseScheduler):
    def __init__(self, query, station):
        super(BaseBarisScheduler, self).__init__(query, station)
        self.baris_data = None
        self.set_event()

    def schedule2json(self, pagination_json, paginated_schedule):
        self.add_extra_data(paginated_schedule)

        json_segments = []
        for flight in paginated_schedule:
            try:
                route_json = self.baris_schedule_route2json(flight)
                json_segments.append(route_json)
            except Exception as ex:
                log.exception('Error while making flight json: {}'.format(ex.message))

        result_json = self.base_json
        result_json.update({
            "pagination": pagination_json,
            "schedule": json_segments,
            "interval_schedule": []
        })

        return result_json

    def add_extra_data(self, paginated_schedule):
        pass

    def baris_schedule_route2json(self, flight):
        raise NotImplementedError

    def base_baris_schedule_route2json(self, flight):
        airline_id = flight.get("airlineID", None)
        if airline_id in self.baris_data.companies_by_ids:
            company_json = Company2Json()(company=self.baris_data.companies_by_ids[airline_id])
        else:
            company_json = None

        model_id = flight.get("transportModelID", None)
        if model_id in self.baris_data.transport_models_by_ids:
            model_json = t_model2json(self.baris_data.transport_models_by_ids[model_id])
        else:
            model_json = None

        title = get_flight_title(self.baris_data, flight, is_one_flight=False, separator="\u2014")

        thread_json = {
            "number": flight["title"],
            "title": title,
            "short_title": title,
            "carrier": company_json,
            "vehicle": model_json,
            "transport_type": "plane",
            "express_type": None,
            "transport_subtype": {"title": None, "code": None, "color": None},
        }

        return {
            "thread": thread_json,
            "terminal": flight["terminal"] if flight["terminal"] else None,
            "is_fuzzy": False,
            "stops": "",
            "platform": "",
            "except_days": None
        }


class OneDayBarisScheduler(BaseBarisScheduler):
    def __init__(self, query, station):
        super(OneDayBarisScheduler, self).__init__(query, station)
        self.show_tablo = query["tablo"] and self.station.tablo_state == "real"

    def get_schedule(self):
        dt_after = datetime.combine(self.date, time(0))
        dt_before = dt_after + timedelta(days=1)
        baris_response = baris.get_station_tablo(
            self.station.id, direction=self.event, after=dt_after, before=dt_before
        )
        self.baris_data = BarisData(baris_response)
        for flight in self.baris_data.flights:
            if flight["datetime"]:
                flight["parsedDatetime"] = parser.parse(flight["datetime"])

        self.baris_data.flights = [
            f for f in self.baris_data.flights
            if f["datetime"] and dt_after <= f["parsedDatetime"].replace(tzinfo=None) < dt_before
        ]
        self.baris_data.flights.sort(key=lambda f: f["parsedDatetime"])

        return self.baris_data.flights

    def add_extra_data(self, paginated_schedule):
        """Добавление дней хождения в рейсы на день"""
        result_timezone = self.result_pytz.zone if self.result_pytz else None
        all_days_response = baris.get_station_all_days_tablo(
            self.station.id, self.station.time_zone, translation.get_language(),
            direction=self.event, result_timezone=result_timezone, run_date=self.date
        )

        flights_by_numbers = {s["title"]: s for s in paginated_schedule}
        for all_days_flight in all_days_response.flights:
            if all_days_flight["title"] in flights_by_numbers:
                flights_by_numbers[all_days_flight["title"]]["daysText"] = all_days_flight[
                    "tzDaysText" if "tzDaysText" in all_days_flight else "daysText"
                ]

    def baris_schedule_route2json(self, flight):
        route_json = self.base_baris_schedule_route2json(flight)

        start_dt_field = "startDatetime" if "startDatetime" in flight and flight["startDatetime"] else "datetime"
        route_json["thread"]["uid"] = make_pseudo_uid_for_baris_flight(
            flight["title"], flight["airlineID"], parser.parse(flight[start_dt_field]).date()
        )

        another_event = "arrival" if self.event == "departure" else "departure"

        route_json.update({
            self.event: (
                flight["parsedDatetime"].astimezone(self.result_pytz) if self.result_pytz else flight["parsedDatetime"]
            ).isoformat(),
            another_event: None,
            "days": flight["daysText"] if "daysText" in flight else ""
        })

        if self.show_tablo:
            status = "cancelled" if flight["status"]["status"] == "cancelled" else None
            str_dt = flight["status"][self.event]
            if not status and str_dt:
                real_dt = parser.parse(str_dt).strftime("%Y-%m-%dT%H:%M:%S")
            else:
                real_dt = None
            terminal = flight["status"]["{}Terminal".format(self.event)]
            if not terminal:
                terminal = None

            route_json["tablo"] = {
                self.event: flight["parsedDatetime"].strftime("%Y-%m-%dT%H:%M:%S"),
                "real_{}".format(self.event): real_dt,
                another_event: None,
                "real_{}".format(another_event): None,
                "terminal": terminal,
                "status": status
            }

        return route_json


class AllDaysBarisScheduler(BaseBarisScheduler):
    def __init__(self, query, station):
        super(AllDaysBarisScheduler, self).__init__(query, station)

    def get_schedule(self):
        result_timezone = self.result_pytz.zone if self.result_pytz else None
        baris_response = baris.get_station_all_days_tablo(
            self.station.id, self.station.time_zone, translation.get_language(),
            direction=self.event, result_timezone=result_timezone
        )
        self.baris_data = BarisData(baris_response)
        self.baris_data.flights.sort(key=lambda f: f["nearestDatetime"].time())

        return self.baris_data.flights

    def baris_schedule_route2json(self, flight):
        route_json = self.base_baris_schedule_route2json(flight)

        route_json["thread"]["uid"] = make_pseudo_uid_for_baris_flight(
            flight["title"], flight["airlineID"], flight["naiveStart"].date()
        )

        another_event = "arrival" if self.event == "departure" else "departure"
        dt = flight["nearestDatetime"].astimezone(self.result_pytz) if self.result_pytz else flight["nearestDatetime"]

        route_json.update({
            self.event: dt.strftime("%H:%M"),
            another_event: None,
            "days": flight["tzDaysText"] if "tzDaysText" in flight else flight["daysText"]
        })

        return route_json
