# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from travel.rasp.library.python.common23.date import environment
from common.utils.date import MSK_TZ
from common.models.schedule import RThread
from common.models_utils import fetch_related
from common.models.transport import TransportSubtype
from stationschedule import station_schedule
from stationschedule.type.base import BaseIntervalSchedule
from stationschedule.models import IntervalScheduleRoute

from travel.rasp.api_public.api_public.v3.station.rasp_db_json_helpers import (
    interval_schedule_route2json, schedule_route2json
)
from travel.rasp.api_public.api_public.v3.station.base_scheduler import BaseScheduler


class RaspDbScheduler(BaseScheduler):
    def __init__(self, query, station, t_type_codes):
        super(RaspDbScheduler, self).__init__(query, station)
        self.direction = query["direction"]
        self.t_type_codes = t_type_codes
        self.next_plan = None
        self.schedule = None

        if self.t_type_codes == ["suburban"]:
            self.schedule_type = "suburban"
        else:
            self.schedule_type = "schedule"
            self.set_event()

    def get_schedule(self):
        kwargs = {}
        if self.schedule_type == "suburban":
            kwargs = dict(requested_direction=self.direction, default_direction="all")

        self.schedule = station_schedule(
            self.station, self.schedule_type, t_type_code=self.t_type_codes,
            event=self.event, limit=None, offset=None, schedule_date=self.date,
            **kwargs
        )

        msk_today = environment.now_aware().astimezone(MSK_TZ).date()
        _, self.next_plan = self.schedule.current_next_plans(msk_today)

        interval_schedule = BaseIntervalSchedule(self.station, t_type_code=self.t_type_codes)
        interval_schedule.build(schedule_date=self.date)

        return interval_schedule.schedule_routes + self.schedule.schedule_routes

    def schedule2json(self, pagination_json, paginated_schedule):
        threads = [schedule_route.thread for schedule_route in paginated_schedule]
        fetch_related(threads, "t_subtype", model=RThread)
        subtypes = [t.t_subtype for t in threads if t.t_subtype]
        fetch_related(subtypes, "color", model=TransportSubtype)

        json_segments = []
        json_interval_segments = []

        for schedule_route in paginated_schedule:
            if isinstance(schedule_route, IntervalScheduleRoute):
                json_interval_segments.append(
                    interval_schedule_route2json(schedule_route, self)
                )
            else:
                json_segments.append(
                    schedule_route2json(schedule_route, self)
                )

        result_json = self.base_json

        result_json.update({
            "pagination": pagination_json,
            "schedule": json_segments,
            "interval_schedule": json_interval_segments
        })

        if self.schedule_type == "suburban":
            result_json.update({
                "directions": [
                    {"code": c, "title": t} for c, t in self.schedule.direction_title_by_code.items()
                ],
                "schedule_direction": {
                    "code": self.schedule.direction_code,
                    "title": self.schedule.direction_title
                }
            })

        return result_json
