# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import time as os_time
from copy import copy
from collections import OrderedDict
from datetime import datetime

from django.core.cache import cache

from common.models.transport import TransportType
from common.utils.caching import global_cache_set, global_cache_add
from common.utils.date import MSK_TZ

tt_plane_id = TransportType.PLANE_ID
tt_train_id = TransportType.TRAIN_ID

JSON_DATE_FMT = '%Y-%m-%d'
# Тип(seat, tariff, ufs), результат (success, timeout, error), взято из кеша(true, false)
# время работы, описание
STAT_INFO = u'%(type)s\t%(result)s\tcached=%(cached)s\t%(time)s\t%(description)s'

# TODO: ROUNDTRIP_PRICE_FORMAT if needed

log = logging.getLogger('rasp.seat_price')
dump_prices_log = logging.getLogger('rasp.order.dump_seat_price')


class Query(object):
    segments = []

    is_order = False
    initial_point_from = None
    initial_point_to = None
    cache_timeout = 1 * 60 * 60,  # по умолчанию 1 час

    def __init__(self, **kwargs):
        for k, v in kwargs.iteritems():
            setattr(self, k, v)

    def copy(self):
        return copy(self)


def segment_data(segment):
    try:
        return segment.data
    except AttributeError:
        return "%s%s" % (segment.departure.strftime("%H%M"), segment.arrival.strftime("%H%M"))


class AllSuppliersTariffInfo(object):
    info_template = {}

    def __init__(self, departure_date, segment=None, segment_key=None):
        if segment and segment_key is None:
            segment_key = self.get_segment_key(segment)

        self.segment_key = segment_key
        self.departure_date = departure_date
        self.by_supplier = {}

    @property
    def has_info(self):
        return bool(self.by_supplier)

    @classmethod
    def get_segment_key(cls, segment):
        if segment.t_type.code != 'bus':
            return segment.number.replace(' ', '-')

        segment_key = segment.number or segment.thread and segment.thread.hidden_number or ''

        # для сопоставления рейсов можно (но аккуратно, только для некоторых партнеров)
        # использовать время отправления в качестве ключа
        if not segment_key and hasattr(segment, 'supplier_code'):
            if segment.supplier_code == 'swdfactory':
                segment_key = segment.departure.strftime('time%H%M')

        return segment_key.replace(' ', '-')

    @property
    def route_key(self):
        return "%s-%s" % (self.segment_key, self.departure_date.strftime("%m%d"))

    def add_ti_info(self, supplier, ti_info):

        assert ti_info is not None

        old_ti_info = self.by_supplier.get(supplier)

        if old_ti_info:
            old_ti_info.update(ti_info)

        else:
            self.by_supplier[supplier] = ti_info


class SupplierReplyTime(object):
    seats = None
    seats_reason = None
    tariffs = None
    tariffs_reason = None
    roundtrip = None
    roundtrip_reason = None
    display_fields = ['seats', 'seats_reason', 'tariffs', 'tariffs_reason',
                      'roundtrip', 'roundtrip_reason']

    def __init__(self, **kwargs):
        for k, v in kwargs.items():
            setattr(self, k, v)

    def __repr__(self):
        result = u"%s:\n" % self.__class__.__name__
        result += u"\n".join(u"%s: %r" % (f, getattr(self, f))
                             for f in self.display_fields)
        return result.encode('utf8')

    @property
    def complete(self):
        return 'timeout' not in [self.seats_reason, self.tariffs_reason, self.roundtrip_reason]

    @property
    def seats_success(self):
        return self.seats if self.seats_reason == 'success' else None


class TariffInfo(object):
    supplier = None
    route_number = None
    seats = None
    tariffs = None
    roundtrip_tariffs = None
    roundtrip_seats = None
    seats_time = None
    tariffs_time = None
    roundtrip_tariffs_time = None
    roundtrip_seats_time = None
    train_info = None
    train_info_time = None
    et_possible = False
    data_fields = ['seats', 'tariffs', 'roundtrip_tariffs', 'roundtrip_seats', 'train_info']
    time_fields = ['seats_time', 'tariffs_time',
                   'roundtrip_tariffs_time', 'roundtrip_seats_time', 'train_info_time']
    out_fields = data_fields

    def __init__(self, route_number, seats=None, tariffs=None,
                 roundtrip_tariffs=None, roundtrip_seats=None):
        self.route_number = route_number
        self.seats = seats
        self.tariffs = tariffs
        self.roundtrip_tariffs = roundtrip_tariffs
        self.roundtrip_seats = roundtrip_seats
        self.order_info = {}

    def update(self, ti_info):
        for field_name in self.data_fields + self.time_fields:
            self_attr = getattr(self, field_name)
            ti_info_attr = getattr(ti_info, field_name)

            if self_attr is None and ti_info_attr is not None:
                setattr(self, field_name, ti_info_attr)

    def set_time(self, lmt):
        for field_name in self.data_fields:
            if getattr(self, field_name) is not None:
                setattr(self, field_name + '_time', lmt)

    def __repr__(self):
        template = u"""<%(class)s:
supplier: %(supplier)s
route_number: %(route_number)s
et_possible: %(et_possible)s
""" + u"\n".join('%s: %%(%s)s' % (f, f) for f in self.out_fields + self.time_fields) + "\n>"
        params = {}
        params['class'] = self.__class__.__name__
        params['supplier'] = self.supplier
        params['route_number'] = self.route_number
        for field_name in self.out_fields:
            params[field_name] = repr(getattr(self, field_name)).decode('utf8', 'ignore')
        for field_name in self.time_fields:
            params[field_name] = repr(getattr(self, field_name)).decode('utf8', 'ignore')
        params['tariffs'] = repr(self.tariffs)
        params['roundtrip_tariffs'] = repr(self.roundtrip_tariffs)
        params['roundtrip_seats'] = repr(self.roundtrip_seats)
        params['et_possible'] = self.et_possible

        for k in params:
            params[k] = unicode(params[k])

        return (template % params).encode('utf8')


class Result(object):
    type = None
    supplier = None
    tariffinfo_class = TariffInfo
    data_types = []
    key_of_delay = None
    query = None

    def __init__(self, query, data, reason, error_text=None, cached=False):
        self.query = query
        self.data = data
        self.reason = reason
        if reason != 'timeout':
            self.lmt = os_time.time()
        else:
            self.lmt = None

        # Добавляем в места цены информацию о поставщике и времени извлечения.
        if reason == 'success' and data:
            for tariff_info in data.values():
                tariff_info.supplier = self.supplier
                tariff_info.set_time(self.lmt)

        self.cached = cached
        self.error_text = error_text

        if reason == 'success':
            self.on_success()

    def __str__(self):
        template = u"""<Result:
class: %(class_name)s
data: %(data)s
reason: %(reason)s
cached: %(cached)s
error_text: %(error_text)s>"""
        params = {}
        params['class_name'] = self.__class__.__name__
        params.update(self.__dict__)
        params['data'] = repr(self.data).decode("utf8")
        return (template % params).encode('utf8')

    def get_key(self):
        return self.key_of_delay + self.query.date.strftime(JSON_DATE_FMT)

    def log_me(self, start, message):
        log.info(STAT_INFO, {
            'type': self.type,
            'result': self.reason,
            'cached': self.cached,
            'time': os_time.time() - start,
            'description': message
        })

    def update_segments(self, segments, reply_time):
        if self.reason == 'success':
            for s in segments:
                ti_info = self.data.get(s.info.segment_key, None)

                if ti_info:
                    s.info.add_ti_info(self.supplier, ti_info)

        for field in self.data_types:
            setattr(reply_time, field, self.lmt)
            setattr(reply_time, field + '_reason', self.reason)

    def on_success(self):
        pass

    def on_error(self, segment):
        pass

    @classmethod
    def get_info_class(cls):
        return cls.tariffinfo_class

    def timeout_or_none_in_cache(self, key):
        result = cache.get(key)
        return result is None or result.reason == 'timeout'

    def set_cache_if_empty(self, key, timeout):
        global_cache_add(key, self, timeout)

    def update_cache(self, key, timeout):
        global_cache_set(key, self, timeout)

    def cache_me(self, key, timeout):
        # В кеше данные должны лежать с флагом cached
        self.cached = True
        if self.timeout_or_none_in_cache(key):
            if self.reason != 'timeout':
                self.update_cache(key, timeout)
            else:
                self.set_cache_if_empty(key, timeout)

            log.info(u'Положили в кэш %s %s на %.3f часа %s', self.type,
                     self.reason, timeout / 3600.0, key)

            if self.reason == 'success' and self.data:
                self.dump_prices()

        else:
            log.warning(u"В кеше %s уже лежат ранее полученные данные",
                        key)
        # Это первое извлечение
        self.cached = False

    def dump_prices(self):
        timestamp = datetime.today()
        timezone = MSK_TZ.localize(timestamp).strftime('%z')
        point_from = self.query.point_from
        point_to = self.query.point_to
        date_forward = self.query.date.strftime('%Y-%m-%d')
        date_backward = None
        for route_number, tariffinfo in self.data.items():
            params = OrderedDict((
                ('tskv_format', 'rasp-tariffs-log'),
                ('timestamp', datetime.today().strftime("%Y-%m-%d %H:%M:%S")),
                ('timezone', timezone),
                ('partner', self.supplier),
                ('type', self.key_of_delay.split(u'_')[0]),
                ('date_forward', date_forward),
                ('date_backward', date_backward),
                ('object_from_type', point_from.__class__.__name__),
                ('object_from_id', point_from.id),
                ('object_from_title', point_from.title),
                ('object_to_type', point_to.__class__.__name__),
                ('object_to_id', point_to.id),
                ('object_to_title', point_to.title),
                ('route_number', route_number)
            ))

            by_class = {}
            if tariffinfo.seats:
                for klass, seats in tariffinfo.seats.items():
                    by_class.setdefault(klass, {})['seats'] = seats

            if tariffinfo.tariffs:
                for klass, tariff in tariffinfo.tariffs.items():
                    by_class.setdefault(klass, {})['tariff'] = tariff.get('price')

            for klass, info in by_class.items():
                if 'seats' in info:
                    params[u"class_" + klass + u"_seats"] = info['seats']
                if 'tariff' in info:
                    params[u"class_" + klass + u"_tariff"] = info['tariff']

            message = u"tskv\t" + u"\t".join(u"%s=%s" % (key, value) for key, value in params.items())

            dump_prices_log.info(message)
