# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import datetime, timedelta
from dateutil import parser

import requests
from django.utils import translation

from common.data_api.baris.instance import baris
from common.data_api.baris.helpers import BarisData, BarisMasksProcessor, get_flight_title
from common.models.geo import Station, Settlement
from common.xgettext.i18n import gettext
from travel.rasp.library.python.common23.date import environment
from common.utils.title_generator import DASH

from travel.rasp.api_public.api_public.v3.core.api_errors import ApiError
from travel.rasp.api_public.api_public.v3.core.helpers import dt_to_tz_date, get_code_getter
from travel.rasp.api_public.api_public.v3.views.json_helpers import Company2Json, t_model2json, point2json


def _get_baris_masks_processor(schedule, stations_by_ids, departure_dt, last_date):
    """
    Если маска расписания содержит указанную дату, то возвращаем процессор маски и ближайшую дату хождения
    :schedule: расписание рейса из БАРиС
    :param stations_by_ids: словарь станций
    :param departure_dt: время отправления, если известно
    :param last_date: последняя дата хождения нитки, используется если не задано departure_dt
    """
    start_station_id = schedule["route"][0]["airportID"]
    if start_station_id in stations_by_ids:
        time_zone = stations_by_ids[start_station_id].time_zone
        processor = BarisMasksProcessor(schedule["masks"], time_zone)

        if departure_dt:
            start_date = dt_to_tz_date(departure_dt, processor.pytz)
            if processor.run_mask[start_date]:
                return processor, start_date
        else:
            if processor.run_mask[last_date]:
                return processor, processor.nearest_run_date

    return None, None


def _build_station_dt(start_date, time_str, station, days_shift=0):
    start_time = parser.parse(time_str).time()
    naive_start_dt = datetime.combine(start_date, start_time)
    naive_start_dt = naive_start_dt + timedelta(days=days_shift)
    return station.pytz.localize(naive_start_dt)


def _check_point(baris_data, point, airport_id):
    if isinstance(point, Station) and airport_id == point.id:
        return True
    if (
        isinstance(point, Settlement)
        and airport_id in baris_data.stations_by_ids
        and baris_data.stations_by_ids[airport_id].settlement_id == point.id
    ):
        return True

    return False


def _get_from_and_to_points_indices(baris_data, route, point_from, point_to):
    if not point_from or not point_to:
        return None, None

    from_index = to_index = -1
    for index in range(0, len(route)):
        airport_id = route[index]["airportID"]
        if _check_point(baris_data, point_from, airport_id):
            from_index = index
        if from_index != -1 and _check_point(baris_data, point_to, airport_id):
            to_index = index

    if from_index == -1:
        raise ApiError(gettext("Нитка не проходит через указанный пункт отправления"), http_code=400)
    if to_index == -1:
        raise ApiError(gettext("Нитка не проходит через указанный пункт прибытия"), http_code=400)

    return from_index, to_index


def _make_stops(baris_data, route, start_date, show_systems, point_from, point_to):
    """Формирование json со списком остановок"""

    from_index, to_index = _get_from_and_to_points_indices(baris_data, route, point_from, point_to)

    start_station = baris_data.stations_by_ids[route[0]["airportID"]]
    start_dt = _build_station_dt(start_date, route[0]["departureTime"], start_station)

    code_getter = get_code_getter(baris_data.stations_by_ids.values(), show_systems)

    result = {
        "stops": [],
        "departure_date": None,
        "arrival_date": None
    }
    for index, route_station in enumerate(route):
        station = baris_data.stations_by_ids[route_station["airportID"]]
        stop_json = {
            "station": point2json(station, code_getter),
            "platform": "",
        }

        departure_dt = None
        if "departureTime" in route_station:
            days_shift = route_station["departureDayShift"] if "departureDayShift" in route_station else 0
            departure_dt = _build_station_dt(start_date, route_station["departureTime"], station, days_shift)
            stop_json["departure"] = departure_dt.strftime("%Y-%m-%d %H:%M:%S")
        else:
            stop_json["departure"] = None

        arrival_dt = None
        if "arrivalTime" in route_station:
            days_shift = route_station["arrivalDayShift"] if "arrivalDayShift" in route_station else 0
            arrival_dt = _build_station_dt(start_date, route_station["arrivalTime"], station, days_shift)
            stop_json["arrival"] = arrival_dt.strftime("%Y-%m-%d %H:%M:%S")
            stop_json["stop_time"] = int((departure_dt - arrival_dt).total_seconds()) if departure_dt else None
        else:
            stop_json["arrival"] = None
            stop_json["stop_time"] = None

        duration_dt = departure_dt if departure_dt else arrival_dt
        stop_json["duration"] = int((duration_dt - start_dt).total_seconds())

        if "departureTerminal" in route_station and route_station["departureTerminal"]:
            stop_json["terminal"] = route_station["departureTerminal"]
        elif "arrivalTerminal" in route_station and route_station["arrivalTerminal"]:
            stop_json["terminal"] = route_station["arrivalTerminal"]
        else:
            stop_json["terminal"] = None

        result["stops"].append(stop_json)

        if index == from_index:
            result["departure_date"] = departure_dt.strftime("%Y-%m-%d")
            now = environment.now_aware().astimezone(station.pytz)
            result["gone"] = departure_dt < now

        if index == to_index:
            result["arrival_date"] = arrival_dt.strftime("%Y-%m-%d")

    return result


def get_baris_thread_json(pseudo_uid, departure_dt, point_from, point_to, show_systems):
    """Формирование данных о нитке из БАРиС"""
    try:
        baris_response = baris.get_flight_schedule(pseudo_uid.flight_number)
    except requests.HTTPError as ex:
        if ex.response.status_code == 404:
            raise ApiError(gettext("Рейса с uid {} нет в базе").format(pseudo_uid.uid), http_code=404)
        raise

    baris_data = BarisData(baris_response)
    flight = baris_data.flights[0]

    result_json = {
        "uid": pseudo_uid.uid,
        "number": flight["title"],
        "express_type": None,
        "transport_type": "plane",
        "transport_subtype": {"color": None, "code": None, "title": None}
    }

    masks_processor = selected_schedule = start_date = None
    for schedule in flight["schedules"]:
        masks_processor, start_date = _get_baris_masks_processor(
            schedule, baris_data.stations_by_ids, departure_dt, pseudo_uid.last_date
        )
        if masks_processor:  # Выбираем из расписаний подходящее под departure_dt или pseudo_uid.last_date
            selected_schedule = schedule
            break

    if not masks_processor:
        if departure_dt:
            raise ApiError(
                gettext("Рейс с uid {} не ходит {}").format(pseudo_uid.uid, departure_dt.strftime("%Y-%m-%d")),
                http_code=404
            )
        raise ApiError(gettext("Рейса с uid {} нет в базе").format(pseudo_uid.uid), http_code=404)

    start_time = parser.parse(selected_schedule["route"][0]["departureTime"])
    title = get_flight_title(baris_data, selected_schedule, is_one_flight=True, separator=DASH)
    result_json.update({
        "start_time": start_time.strftime("%H:%M"),
        "start_date": start_date.strftime("%Y-%m-%d"),
        "days": masks_processor.get_days_text(translation.get_language()),
        "except_days": "",
        "title": title,
        "short_title": title
    })

    company = baris_data.companies_by_ids.get(flight.get("airlineID"))
    result_json["carrier"] = Company2Json()(company) if company else None

    transport_model = baris_data.transport_models_by_ids.get(selected_schedule.get("transportModelID"))
    result_json["vehicle"] = t_model2json(transport_model) if transport_model else None

    if point_from and point_to:
        code_getter = get_code_getter((point_from, point_to), show_systems)
        result_json["from"] = point2json(point_from, code_getter)
        result_json["to"] = point2json(point_to, code_getter)
    else:
        result_json["from"] = result_json["to"] = None

    result_json.update(_make_stops(
        baris_data, selected_schedule["route"], start_date, show_systems, point_from, point_to
    ))

    return result_json
