# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import operator
from datetime import datetime

from common.models.geo import CompanyOffice
from common.models.schedule import TrainSchedulePlan
from travel.rasp.library.python.common23.date import environment
from common.views.thread import get_interval_string

from travel.rasp.api_public.api_public.v3.views.json_helpers import (
    Thread2Json, point2json, tickets_info2json, companyoffice2json
)
from travel.rasp.api_public.api_public.v3.core.helpers import get_code_getter, get_currency_info, find_rtstation
from travel.rasp.api_public.api_public.v3.thread.rasp_db_thread import get_thread_segment


def threadinfo2json(thread, path, naive_start_dt, show_systems, next_plan=None, result_timezone=None):
    first_rtstation = path[0]
    thread_start_dt = thread.pytz.localize(naive_start_dt)
    shift = first_rtstation.calc_days_shift(event="departure",
                                            start_date=thread_start_dt.date(),
                                            event_tz=result_timezone)
    days_text_dict = thread.L_days_text_dict(shift,
                                             thread_start_dt.date(),
                                             next_plan=next_plan)

    json_thread = Thread2Json()(thread)
    json_thread.update({
        "days": days_text_dict.get("days_text", ""),
        "except_days": days_text_dict.get("except_days_text", ""),
        "stops": rtstation_path2json(path, naive_start_dt, thread.pytz, show_systems)
    })

    return json_thread


def rtstation_path2json(path, naive_start_dt, thread_tz, show_systems):
    stations_to_collect = map(operator.attrgetter("station"), path)

    code_getter = get_code_getter(stations_to_collect, show_systems)

    start_dt = thread_tz.localize(naive_start_dt)

    rtstations = []

    for rts in path:
        if rts.is_no_stop():
            continue  # show stations with stops only [RASPFRONT-9975]

        departure_dt = None
        if rts.tz_departure is not None:
            departure_dt = rts.get_departure_loc_dt(naive_start_dt)

        arrival_dt = None
        if rts.tz_arrival is not None:
            arrival_dt = rts.get_arrival_loc_dt(naive_start_dt)

        stop_time = None
        if arrival_dt is not None and departure_dt is not None:
            stop_time = int((departure_dt - arrival_dt).total_seconds())

        duration = 0
        if departure_dt:
            duration = (departure_dt - start_dt).total_seconds()
        elif arrival_dt:
            duration = (arrival_dt - start_dt).total_seconds()

        rtstations.append({
            "station": point2json(rts.station, code_getter),
            "departure": departure_dt,
            "arrival": arrival_dt,
            "duration": duration,
            "stop_time": stop_time,
            "platform": rts.L_platform(),
            "terminal": rts.terminal_id and rts.terminal.name
        })

    return rtstations


def get_common_thread_json(thread, path, naive_start_dt, point_from, point_to, national_version=None,
                           tld=None, show_systems=None, currency=None, client_city=None, result_timezone=None):
    current_plan, next_plan = TrainSchedulePlan.add_to_threads([thread], environment.today())
    json_thread = threadinfo2json(thread, path, naive_start_dt, show_systems, next_plan=next_plan,
                                  result_timezone=result_timezone)
    local_start_dt = path[0].get_departure_loc_dt(naive_start_dt)
    if result_timezone:
        local_start_dt = local_start_dt.astimezone(result_timezone)

    json_thread.update({
        "from": None,
        "to": None,
        "start_date": local_start_dt.strftime("%Y-%m-%d"),
    })

    if point_from and point_to:
        currency_info = get_currency_info(tld, currency)
        code_getter = get_code_getter((point_from, point_to), show_systems)

        json_thread.update({
            "from": point2json(point_from, code_getter),
            "to": point2json(point_to, code_getter)
        })

        segment = get_thread_segment(national_version, thread, point_from, point_to,
                                     path, naive_start_dt, currency_info, client_city)

        json_thread["gone"] = getattr(segment, "gone", False)

        if segment:
            json_thread["tickets_info"] = tickets_info2json(segment, currency_info)
        else:
            json_thread["tickets_info"] = None

    if thread.company:
        json_thread["carrier"]["offices"] = []
        offices = CompanyOffice.objects.filter(company=thread.company)
        if offices:
            points_to_collect = []
            for office in offices:
                points_to_collect.append(office.settlement)
                points_to_collect.append(office.main_station)

            code_getter = get_code_getter(points_to_collect, show_systems)
            json_offices = [companyoffice2json(office, code_getter) for office in offices]
            json_thread["carrier"]["offices"] = json_offices

    return json_thread, local_start_dt


def get_thread_json(thread, path, naive_start_dt, point_from, point_to, national_version=None,
                    tld=None, show_systems=None, currency=None, client_city=None, result_timezone=None):
    json_thread, local_start_dt = get_common_thread_json(
        thread, path, naive_start_dt, point_from, point_to, national_version,
        tld, show_systems, currency, client_city, result_timezone
    )

    json_thread.update({
        "arrival_date": None,
        "departure_date": None,
        "start_time": local_start_dt.time()
    })

    if point_from and point_to:
        departure_loc_dt = find_rtstation(point_from, path).get_departure_loc_dt(naive_start_dt)
        arrival_loc_dt = find_rtstation(point_to, path).get_arrival_loc_dt(naive_start_dt)

        if result_timezone:
            departure_loc_dt = departure_loc_dt.astimezone(result_timezone)
            arrival_loc_dt = arrival_loc_dt.astimezone(result_timezone)

        json_thread["departure_date"] = departure_loc_dt.strftime("%Y-%m-%d")
        json_thread["arrival_date"] = arrival_loc_dt.strftime("%Y-%m-%d")

    return json_thread


def get_interval_thread_json(thread, path, naive_start_dt, point_from, point_to, national_version=None,
                             tld=None, show_systems=None, currency=None, client_city=None,
                             result_timezone=None, query_dt=None):
    json_thread, local_start_dt = get_common_thread_json(
        thread, path, naive_start_dt, point_from, point_to, national_version,
        tld, show_systems, currency, client_city, result_timezone
    )
    json_thread.update({
        "interval": {
            "density": get_interval_string(thread)
        }
    })

    if query_dt:
        json_thread["interval"].update({
            "begin_time": datetime.combine(naive_start_dt, thread.begin_time).isoformat(),
            "end_time": datetime.combine(naive_start_dt, thread.end_time).isoformat()
        })
    else:
        json_thread["interval"].update({
            "begin_time": thread.begin_time,
            "end_time": thread.end_time
        })

    return json_thread
