# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import operator
from datetime import datetime

from common.models_utils import fetch_related
from common.models.geo import Station
from common.views.tariffs import DisplayInfo
from common.models.schedule import RThread
from common.models_utils.i18n import RouteLTitle
from travel.rasp.library.python.common23.date import environment
from common.xgettext.i18n import gettext
from route_search.models import RThreadSegment

from travel.rasp.api_public.api_public.v3.core.api_errors import ApiError
from travel.rasp.api_public.api_public.v3.core.helpers import dt_to_tz_date
from travel.rasp.api_public.api_public.v3.core import api_errors
from travel.rasp.api_public.api_public.v3.tariffs import bus
from travel.rasp.api_public.api_public.v3.tariffs.views import add_station_station_tariffs, add_dynamic_tariffs
from travel.rasp.api_public.api_public.v3.tariffs.utils import rebase_tariffs


def get_thread(thread_uid, departure_dt):
    try:
        thread = RThread.objects.exclude(
            route__supplier__exclude_from_external_api=True).get(uid=thread_uid)
    except RThread.DoesNotExist:
        raise ApiError(gettext(u"Рейса с uid {} нет в базе").format(thread_uid), http_code=404)

    today = environment.today()
    path = list(thread.display_path)

    if not path:
        raise ApiError(gettext(u"Рейса с uid {} нет в базе").format(thread_uid), http_code=404)

    first_rtstation = path[0]

    RouteLTitle.fetch([thread.L_title])

    first_run_d = thread.first_run(today)
    if not first_run_d:
        raise ApiError(gettext(u"Рейса с uid {} нет в базе").format(thread_uid), http_code=404)

    naive_start_dt = None
    if departure_dt:
        departure_date = dt_to_tz_date(departure_dt, first_rtstation.station.pytz)
        start_date = first_rtstation.calc_thread_start_date(event="departure", event_date=departure_date)

        naive_start_dt = datetime.combine(start_date, thread.tz_start_time)

        if not thread.runs_at(naive_start_dt):
            raise ApiError(gettext(u"Рейс {} не ходит {}").format(thread_uid, departure_date), http_code=404)

    if not naive_start_dt:
        naive_start_dt = datetime.combine(first_run_d, thread.tz_start_time)

    return thread, path, naive_start_dt


def get_thread_segment(national_version, thread, point_from, point_to, path, naive_start_dt, currency_info,
                       client_city):
    fetch_related(
        map(operator.attrgetter("station"), path),
        "settlement",
        model=Station
    )

    try:
        rtstation_from = filter(lambda rts: point_from in (rts.station, rts.station.settlement), path)[0]
    except IndexError:
        raise api_errors.ApiError(gettext(u"Нитка не проходит через указанный пункт отправления"), http_code=400)

    try:
        rtstation_to = filter(lambda rts: point_to in (rts.station, rts.station.settlement), path)[0]
    except IndexError:
        raise api_errors.ApiError(gettext(u"Нитка не проходит через указанный пункт прибытия"), http_code=400)

    departure = rtstation_from.get_loc_departure_dt(naive_start_dt)
    if not departure or rtstation_from == rtstation_to:
        return

    segment = RThreadSegment()
    segment.station_from = rtstation_from.station
    segment.station_to = rtstation_to.station
    segment.thread = thread
    segment.departure = departure
    segment.arrival = rtstation_to.get_arrival_loc_dt(naive_start_dt)
    segment.now = environment.now_aware()
    segment._init_data()
    segment.display_info = DisplayInfo()

    segments = [segment]

    tcode = thread.t_type.code
    if tcode == "suburban" or (national_version == "tr" and tcode == "train"):
        add_station_station_tariffs(segments)

        return segments[0]

    elif tcode in ("plane", "train", "bus"):
        add_dynamic_tariffs(segments, [tcode], point_from, point_to, naive_start_dt.date(), client_city,
                            national_version)

        if tcode == "bus":
            bus.add_tariffs(segments)

    else:  # Водный тип транспорта
        bus.add_tariffs(segments)

    rebase_tariffs(segments, currency_info.rates)

    return segments[0]
