# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from django.conf import settings
from django.utils.encoding import smart_unicode
from rest_framework.decorators import APIView

from common.xgettext.i18n import gettext

from travel.rasp.api_public.api_public.v3.core.api_errors import ApiError, ApiInternalError
from travel.rasp.api_public.api_public.v3.core.api_response import build_response
from travel.rasp.api_public.api_public.v3.views.error_view import error_view, gen_response_error_message

log = logging.getLogger(__name__)


class BaseApiView(APIView):
    http_method_names = ['get']

    def http_method_not_allowed(self, request, *args, **kwargs):
        error = ApiError(gettext(u'Использован неподдерживаемый тип запроса - {}. '
                                 u'Допустимые типы: {}.'.format(request.method, u''.join(self.http_method_names).upper())), 400)
        return error_view(request, error)

    def get(self, request, *args, **kwargs):
        return self.make_response(self.handle, request, *args, **kwargs)

    def post(self, request, *args, **kwargs):
        return self.make_response(self.handle, request, *args, **kwargs)

    def handle(self, *args, **kwargs):
        raise NotImplementedError()

    def make_response(self, handle, request, *args, **kwargs):
        http_code = kwargs.get('http_code', 200)
        try:
            result = handle(request, *args, **kwargs)
            if not isinstance(result, (dict, basestring)):
                raise ApiInternalError(u'Неправильный тип ответа: {}\n{}\n'.format(
                    smart_unicode(repr(result)),
                    smart_unicode(repr(request))
                ))
        except ApiError as ex:
            result, http_code = gen_response_error_message(request, ex)
        except Exception as ex:
            if settings.DEBUG:
                raise

            log.error(
                u'Неизвестная ошибка: %s\n%s\n', repr(ex), smart_unicode(repr(request)),
                exc_info=True,
                extra={'request': request}  # for raven
            )
            result, http_code = {'error': {'text': gettext(
                u'Сервис Яндекс.Расписания временно недоступен.'
                u' Мы уже в курсе данной проблемы и работаем над ее устранением.'
                u' Приносим свои извинения за неудобства.')}}, 500

        return build_response(request, result, http_code)
