# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import itertools
import operator

from common.models.geo import CompanyOffice
from common.models.schedule import Company
from common.xgettext.i18n import gettext

from travel.rasp.api_public.api_public.v3.core import points
from travel.rasp.api_public.api_public.v3.core.api_errors import ApiError
from travel.rasp.api_public.api_public.v3.core.helpers import get_code_getter
from travel.rasp.api_public.api_public.v3.views.base_api_view import BaseApiView
from travel.rasp.api_public.api_public.v3.views.json_helpers import Company2Json
from travel.rasp.api_public.api_public.v3.request_schemas import CarrierRequestSchema


class CarrierView(BaseApiView):
    def handle(self, request, *args, **kwargs):
        query, errors = CarrierRequestSchema(context={'request': request}).load(request.GET)

        carriers = get_carriers(query['system'], query['code'])
        offices = CompanyOffice.objects.filter(company__in=carriers)

        if query['city']:
            settlement = points.find_point(query['system'], query['city'])
            offices = offices.filter(settlement=settlement)

        offices = offices.order_by('company', 'settlement')
        company_offices_dict = {cid: list(comp_offices)
                                for cid, comp_offices
                                in itertools.groupby(offices, operator.attrgetter('company_id'))}

        points_to_collect = []
        for office in offices:
            points_to_collect.append(office.main_station)
            points_to_collect.append(office.settlement)

        code_getter = get_code_getter(points_to_collect, query['show_systems'])
        carriers = map(lambda c: Company2Json(verbose=True)(c, company_offices_dict.get(c.id, []), code_getter),
                       carriers)

        if len(carriers) == 1:
            return {
                'carrier': carriers[0]
            }

        return {
            'carriers': carriers
        }


def get_carriers(system, carrier_code):
    carriers = []

    if system == 'yandex':
        try:
            carrier_code = int(carrier_code)
            carriers = [Company.hidden_manager.get(id=carrier_code)]
        except (Company.DoesNotExist, ValueError):
            pass

    elif system == 'iata':
        carriers = list(Company.hidden_manager.filter(iata=carrier_code))

    elif system == 'icao':
        try:
            carriers = [Company.hidden_manager.get(icao=carrier_code)]
        except Company.DoesNotExist:
            pass

    elif system == 'sirena':
        try:
            carriers = [Company.hidden_manager.get(sirena_id=carrier_code)]
        except Company.DoesNotExist:
            pass

    if not carriers:
        raise ApiError(gettext(u'Не нашли компании по коду {system}="{code}"').format(
            system=system, code=carrier_code
        ), http_code=404)

    carriers.sort(key=lambda c: (c.priority, c.id))

    return carriers
