# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.utils import translation

from common.models.geo import Station
from travel.rasp.api_public.api_public.v3.tariffs import currency


def point2json(point, code_getter=None, add_coords=False):
    if not point:
        return

    json_point = {
        'type': point.type,
        'title': point.L_title(),
        'short_title': point.L_short_title(fallback=False),
        'popular_title': point.L_popular_title(fallback=False),
        'code': point.point_key
    }

    if add_coords:
        json_point['lat'] = point.latitude
        json_point['lng'] = point.longitude

    if code_getter:
        codes = code_getter.get_codes(point)
        if codes:
            json_point['codes'] = codes

    if isinstance(point, Station):
        json_point['station_type'] = point.station_type.code
        json_point['station_type_name'] = point.station_type.L_title()
        json_point['transport_type'] = point.t_type.code

    return json_point


def t_model2json(t_model):
    lang = translation.get_language()
    if t_model:
        if lang in ['ru', 'uk']:
            title = t_model.title or t_model.title_en
        else:
            title = t_model.title_en or t_model.title

        return title


def get_dt_str(dt):
    if dt:
        return dt.isoformat()


def companyoffice2json(office, code_getter=None):
    return {
        'title': office.title,
        'city': point2json(office.settlement, code_getter),
        'main_station': point2json(office.main_station, code_getter=code_getter),
        'address': office.address,
        'contact_info': office.contact_info,
        'phone': office.phone,
        'phone_booking': office.phone_booking,
        'description': office.description,
        'lat': office.latitude,
        'lng': office.longitude,
        'is_main': office.is_main
    }


class Company2Json(object):
    def __init__(self, verbose=False, logo_svg=False):
        self.logo_svg = logo_svg
        self.verbose = verbose

    def __call__(self, company, offices=None, code_getter=None):
        if company:
            json_company = {
                'code': company.id,
                'title': company.L_title(),
                'codes': {
                    'sirena': company.sirena_id,
                    'iata': company.iata,
                    'icao': company.icao
                }
            }

            if self.verbose:
                json_company['address'] = company.address
                json_company['url'] = company.url
                json_company['email'] = company.email
                json_company['contacts'] = company.contact_info
                json_company['phone'] = company.phone
                json_company['logo'] = company.logo.url if company.logo else None

            if self.logo_svg:
                json_company['logo_svg'] = company.svg_logo.url if company.svg_logo else None

            if code_getter:
                json_company['offices'] = [companyoffice2json(office, code_getter) for office in offices]

            return json_company


class Thread2Json(object):
    def __init__(self, company2json=None):
        self.company2json = Company2Json() if company2json is None else company2json

    def __call__(self, thread):
        result = {
            'number': thread.number,
            'title': thread.L_title(),
            'short_title': thread.L_title(short=True),
            'express_type': thread.express_type,
            'transport_type': thread.t_type.code,
            'carrier': self.company2json(thread.company),
            'uid': thread.uid,
            'vehicle': t_model2json(thread.t_model)
        }

        if thread.t_subtype:
            result['transport_subtype'] = {
                'title': thread.t_subtype.L_title(),
                'code': getattr(thread.t_subtype, 'code', None),
                'color': getattr(thread.t_subtype.color, 'color', None)
            }
        else:
            result['transport_subtype'] = {'title': None, 'code': None, 'color': None}

        return result


def tickets_info2json(segment, currency_info):
    if segment.gone:
        return

    tariffs_info = segment.display_info.tariffs_info

    if not tariffs_info:
        return

    json_places = []

    for place in tariffs_info.places:
        price_value, curr = currency.get_price_currency(place.tariff, currency_info)

        json_places.append({
            'name': place.name,
            'price': price_value,
            'currency': curr
        })

    return {
        'et_marker': tariffs_info.et_marker,
        'places': json_places
    }
