# -*- coding: utf-8 -*-

import datetime
import logging

from django.conf import settings

from common.data_api.ticket_daemon import query as ticket_daemon_query
from common.data_api.ticket_daemon.jsend import InvalidResponse, Fail, Error
from common.models.geo import Country
from common.models.partner import Partner
from common.models.transport import TransportType
from travel.rasp.library.python.common23.date import environment
from common.utils.date import KIEV_TZ, MSK_TZ

from travel.rasp.api_public.tariffs.retrieving.base import (
    Result, AllSuppliersTariffInfo, segment_data, SupplierReplyTime, Query
)


daemon_api_query_all_log = logging.getLogger('rasp_api.daemon_api_query_all')


class DaemonResult(Result):
    data_types = ['seats', 'tariffs']

    def dump_prices(self):
        pass

    def __init__(self, supplier, query, data, segments, reason):
        self.supplier = supplier
        self.segments = segments

        Result.__init__(self, query, data, reason)

    def update_segments(self, segments, reply_time):
        if self.reason == 'success':
            self.update_extra_segments(segments)

        for field in self.data_types:
            setattr(reply_time, field, self.lmt)
            setattr(reply_time, field + '_reason', self.reason)

    def update_extra_segments(self, segments):
        self.extra_segments = self.segments.copy()

        def get_basic_key(key):
            return key

        basic_keys = dict(
            (get_basic_key(key), key)
            for key, value in self.extra_segments.items()
        )

        for s in segments:
            route_key = s.info.route_key

            self_segment = self.segments.get(route_key)

            if self_segment:
                ti_info = self.data.get(s.info.segment_key)

                if ti_info:
                    if s.info.has_info and 'bus' in ti_info.tariffs:
                        ti_info.tariffs['bus']['from'] = True

                    s.info.add_ti_info(self.supplier, ti_info)

            full_key = basic_keys.get(get_basic_key(route_key))

            if full_key and full_key in self.extra_segments:
                del self.extra_segments[full_key]

    @classmethod
    def make_results(cls, request, query, user_settlement, initiate_query=False):
        tickets_query = make_tickets_query(query, user_settlement, request.NATIONAL_VERSION)

        if initiate_query:
            try:
                tickets_query.query_all(ignore_cache=settings.IGNORE_DAEMON_CACHE)
            except (InvalidResponse, Fail, Error) as ex:
                daemon_api_query_all_log.exception(u'%s', repr(ex))
                return []
            except Exception as ex:
                daemon_api_query_all_log.exception(u'Unknown error: %s', repr(ex))
                return []

        try:
            variants, _statuses = tickets_query.collect_variants()
        except Exception:
            return []

        def raspfilter(v):
            return False if len(v.forward.segments) != 1 else True

        results = []
        for partner_code, p_variants in variants.items():
            p_variants = filter(raspfilter, p_variants)
            result = cls.make_result(partner_code, query, p_variants)
            results.append(result)

        return results

    @classmethod
    def make_result(cls, partner_code, query, variants):
        ti_class = cls.get_info_class()

        segments = {}
        data = {}

        for variant in variants:
            forward = variant.forward.segments[0]

            if not forward.departure or not forward.arrival:
                continue

            seats, tariffs = cls.make_seats_and_tariffs(variant, forward)

            ti = ti_class(forward.number)

            ti.seats = seats
            ti.tariffs = tariffs
            ti.et_possible = getattr(forward, 'electronic_ticket', False)

            forward.info = AllSuppliersTariffInfo(query.date, forward)

            data[forward.info.segment_key] = ti

            forward.info.add_ti_info(partner_code, ti)

            segments[forward.info.route_key] = forward

        return cls(partner_code, query, data, segments, 'success')

    @classmethod
    def add_tariff_and_seat(cls, request, query, user_settlement=None):
        return cls.make_results(
            request, query, user_settlement,
            initiate_query=settings.QUERY_TICKET_DAEMON
        )

    @classmethod
    def make_seats_and_tariffs(cls, variant, forward):
        seats = {cls.ticket_cls: getattr(variant, 'seats', 1)}
        tariffs = {cls.ticket_cls: variant.order_data}

        tariff_info = variant.order_data
        tariff_info.update({'price': variant.tariff})

        if forward.supplier_code:
            tariff_info['supplier'] = forward.supplier_code

        return seats, tariffs


class PlaneResult(DaemonResult):
    key_of_delay = 'plane_'
    ticket_cls = 'economy'

    def update_extra_segments(self, segments):
        self.extra_segments = self.segments.copy()

        for s in segments:
            route_key = s.info.route_key

            self_segment = self.segments.get(route_key)

            # если нашли соответствующий сегмент из базы
            # и времена прибытия и отправления совпадают
            if self_segment and segment_data(self_segment) == segment_data(s):
                ti_info = self.data.get(s.info.segment_key)

                if ti_info:
                    s.info.add_ti_info(self.supplier, ti_info)

                if route_key in self.extra_segments:
                    del self.extra_segments[route_key]


class BusResult(DaemonResult):
    key_of_delay = 'bus_'
    ticket_cls = 'bus'


def make_tickets_query(query, user_settlement, national_version):
    q = ticket_daemon_query.Query(
        user_settlement=user_settlement,
        point_from=query.point_from,
        point_to=query.point_to,
        date_forward=query.date,
        date_backward=None,
        passengers={'adults': 1},
        klass='economy',
        national_version=national_version,
        t_code=query.t_type.code
    )
    return q


def tariffs_date(point_from, departure, t_type):
    if t_type in ['plane']:
        return departure.date()

    if t_type == 'train':
        if point_from.country_id == Country.UKRAINE_ID:
            return departure.astimezone(KIEV_TZ).date()
        else:
            return departure.astimezone(MSK_TZ).date()

    return departure.date()


def segment_tariffs_date(segment):
    return tariffs_date(segment.station_from, segment.departure, segment.t_type.code)


def add_availability_info(request, segments, point_from, point_to, tcodes=tuple(),
                          early_border=None, late_border=None, user_settlement=None):

    if not settings.SEATS_FETCH_ENABLED:
        return

    plane_groups = {}
    bus_groups = {}

    fill_groups_from_segments(segments, plane_groups, bus_groups)

    fill_groups_from_supplement(tcodes, point_from, point_to, early_border,
                                late_border, plane_groups, bus_groups)

    results = []

    # plane_results = get_plane_results(request, plane_groups, point_from, point_to, user_settlement)
    plane_results = []

    results.extend(plane_results)

    # bus_results = get_bus_results(request, bus_groups, point_from, point_to, user_settlement)
    bus_results = []

    results.extend(bus_results)

    reply_info_by_key = {}

    for result in results:
        key = result.get_key()

        supplier = result.supplier

        key_supplier_time = reply_info_by_key.setdefault(key, {})

        reply_time = key_supplier_time.setdefault(supplier, SupplierReplyTime())

        result.update_segments(segments, reply_time)


def fill_groups_from_segments(segments, plane_groups, bus_groups):
    for segment in segments:
        msk_departure_date = segment.departure.astimezone(MSK_TZ).date()

        departure_date = segment_tariffs_date(segment)

        if segment.t_type.code == 'plane':
            plane_groups.setdefault((msk_departure_date, departure_date), []).append(segment)

        if segment.t_type.code == 'bus':
            bus_groups.setdefault((msk_departure_date, departure_date), []).append(segment)

        segment.info = AllSuppliersTariffInfo(departure_date, segment)


def fill_groups_from_supplement(supplement, point_from, point_to,
                                early_border, late_border,
                                plane_groups, bus_groups):
    for t_type in supplement:
        msk_departure_date = early_border.date()
        departure_date = tariffs_date(point_from, early_border, t_type)
        end_date = tariffs_date(point_to, late_border, t_type)

        while departure_date <= end_date:
            if t_type == 'plane':
                plane_groups.setdefault((msk_departure_date, departure_date), [])

            if t_type == 'bus':
                bus_groups.setdefault((msk_departure_date, departure_date), [])

            departure_date += datetime.timedelta(days=1)


def get_plane_results(request, plane_groups, point_from, point_to, user_settlement):

    results = []

    local_today = point_from.localize(msk=environment.now()).date()

    enabled_partners_full = Partner.get_actual('plane', request.NATIONAL_VERSION, from_rasp=True)

    for (msk_dep_date, departure_date), plane_segments in plane_groups.items():

        enabled_partners = [p for p in enabled_partners_full if p.t_type.code == 'plane']
        if not enabled_partners:
            continue

        # Местная текущая дата-время в городе из которого выезжаем
        # Местная дата-время планируемого выезда
        days_from_today = (departure_date - local_today).days

        # Если это уже в далеком прошлом не получаем данные
        if is_the_plane_distant_past(days_from_today):
            continue

        query = Query(segments=plane_segments, point_from=point_from, point_to=point_to,
                      date=departure_date, t_type=TransportType.objects.get(code='plane'))

        res = PlaneResult.add_tariff_and_seat(request, query, user_settlement)

        results.extend(res)

    return results


def get_bus_results(request, groups, point_from, point_to, user_settlement):
    results = []

    local_today = point_from.localize(msk=environment.now()).date()

    for (msk_dep_date, departure_date), bus_segments in groups.items():
        # Местная текущая дата-время в городе из которого выезжаем
        # Местная дата-время планируемого выезда
        days_from_today = (departure_date - local_today).days

        # Если это уже в далеком прошлом, не получаем данные
        if is_the_bus_distant_past(days_from_today):
            continue

        query = Query(
            segments=bus_segments, point_from=point_from, point_to=point_to,
            date=departure_date, t_type=TransportType.objects.get(code='bus'),
        )

        res = BusResult.add_tariff_and_seat(request, query, user_settlement)

        results.extend(res)

    return results


def get_plane_cache_timeout(days_from_today):
    # 6 часов для рейсов на даты, отстоящие от текущей более чем на 2 месяца,
    if days_from_today > 60:
        cache_timeout = 6 * 60 * 60
    # 3 часа для рейсов на даты, отстоящие от текущей более чем на 1 месяц,
    elif days_from_today > 30 and days_from_today <= 60:
        cache_timeout = 3 * 60 * 60
    # 1 час для рейсов на даты, отстоящие от текущей более чем на 1 неделю,
    elif days_from_today > 7 and days_from_today <= 30:
        cache_timeout = 60 * 60
    # 30 минут для рейсов на даты, отстоящие от текущей более чем на 4 дня,
    elif days_from_today > 4 and days_from_today <= 7:
        cache_timeout = 30 * 60
    # 15 минут для всех остальных.
    else:
        cache_timeout = 15 * 60

    return cache_timeout


def is_the_plane_distant_past(days_from_today):
    return days_from_today < -settings.SEATS_IN_PAST_DAY


def is_the_bus_distant_past(days_from_today):
    return days_from_today < -settings.SEATS_IN_PAST_DAY
