# -*- coding: utf-8 -*-

import itertools
import operator
from datetime import datetime, time, timedelta

from django.db.models import Q
from django.utils.http import urlencode

from common.models.currency import Price
from common.models.tariffs import CLS_MAPPING, AeroexTariff
from travel.rasp.api_public.tariffs.retrieving import add_availability_info


class CookedInfo(object):
    dynamic = True

    def __init__(self, t_type, departure_date):
        self.places = []
        self.t_type = t_type
        self.departure_date = departure_date
        self.can_buy = False
        self.et_marker = False
        self.min_tariff = None
        self.has_data = True
        self.yamoney = False
        self.timestamp = None
        self.no_data_no_seats = False
        self.wait = False

    def copy(self):
        new = CookedInfo(self.t_type, self.departure_date)
        new.can_buy = self.can_buy
        new.et_marker = self.et_marker
        new.min_tariff = self.min_tariff
        new.has_data = self.has_data
        new.yamoney = self.yamoney
        new.timestamp = self.timestamp
        new.no_data_no_seats = self.no_data_no_seats

        return new

    def __json__(self):
        return {
            'places': [p.json() for p in self.places],
            'et_marker': self.et_marker,
        }

    @property
    def request_key(self):
        return '%s_%s' % (self.t_type, self.departure_date)

    @property
    def ndns_class(self):
        if self.no_data_no_seats:
            return ' js-ndns-%s' % self.request_key

        return ''


class TariffPlace(object):
    """Тариф для отображения"""

    def __init__(self, cooked, code, name, type_, timestamp,
                 tariff=None, min_=False):

        self.cooked = cooked
        self.code = code
        self.name = name
        self.type_ = type_
        self.tariff = tariff if hasattr(tariff, 'currency') else Price(tariff)
        self.min = min_
        self.timestamp = timestamp

    def json(self):
        tariff = {
            'value': self.tariff.value,
            'currency': self.tariff.currency,
        }

        return [self.name, self.code, tariff, self.min]


class TariffChoice(TariffPlace):
    def __init__(self, order_data, cooked, code, name, type_, timestamp, tariff=None, min_=False):
        self.order_data = order_data

        TariffPlace.__init__(self, cooked, code, name, type_, timestamp, tariff, min_)

    def link(self, request, segment, buy_link_func, point_from=None, point_to=None):
        order_data = self.order_data.copy()

        order_data.update({
            'tariff': self.tariff.value,
            'currency': self.tariff.currency,
        })

        # Импортируем здесь, так как весь этот класс - хак
        from common.utils.order_data import signed_order_data

        params = {'choice': urlencode(signed_order_data(order_data))}

        if segment:
            return self.cooked.link(request, segment, buy_link_func, point_from, point_to, params)

        return '?' + urlencode(params)


class ClsInfo(object):
    def __init__(self, seats, tariff, timestamp, order_data=None):
        self.seats = seats
        self.tariff = tariff
        self.timestamp = timestamp
        self.order_data = order_data

    def __repr__(self):
        return "<ClsInfo seats=%r tariff=%r timestamp=%r>" % (self.seats, self.tariff, self.timestamp)

    def places(self, cooked, code, name):
        if self.seats and self.tariff:
            if self.order_data:
                yield TariffChoice(
                    self.order_data,
                    cooked, code, name, 'price',
                    self.timestamp,
                    self.tariff['price'],
                    self.tariff.get('from'),
                )

            else:
                yield TariffPlace(
                    cooked, code, name, 'price',
                    self.timestamp,
                    self.tariff['price'],
                    self.tariff.get('from'),
                )


def process_info(raw):
    by_cls = {}

    for supplier, info in raw.by_supplier.items():
        seats = info.seats
        tariffs = info.tariffs
        seats_time = info.seats_time
        tariffs_time = info.tariffs_time

        if not seats:
            continue

        for code, available in seats.items():
            tariff = tariffs and tariffs.get(code)

            # При наличии мест обязательно нужны тарифы
            if available:
                if tariff:
                    # И учитываем время приход тарифов
                    timestamp = tariffs_time

                else:
                    # При отсутствии тарифов игнорируем информацию
                    continue

            else:
                # При отсутствии мест важно только время прихода этой информации
                timestamp = seats_time

            try:
                current_tariff = by_cls[code].tariff['price']

            except KeyError:
                current_tariff = None

            # Записываем данные первого полностью ответившего поставщика
            if current_tariff is None or tariff['price'] < current_tariff:
                order_data = None

                if supplier == 'biletall':
                    url = tariff.get('url')

                    if url:
                        order_data = {
                            'url': url,
                            'partner': supplier,
                            'ucus': tariff['ucus'],
                        }

                by_cls[code] = ClsInfo(available, tariff, timestamp, order_data)

    return by_cls


def cook_tariffs(raw, t_type):
    """
    Подготовка данных с информацией о местах и ценах для отображения на странице
    """

    cooked = CookedInfo(t_type, raw.departure_date)

    by_cls = process_info(raw)

    # Преобразовываем словарь в пригодный для отображения
    for name, code in CLS_MAPPING:
        if code in by_cls:
            cooked.places.extend(by_cls[code].places(cooked, code, name))

    return cooked


def add_dynamic_tariffs(segments, tcodes, point_from, point_to, when, request):
    segments = list(itertools.ifilterfalse(operator.attrgetter('gone'), segments))

    add_availability_info(request, segments, point_from,
                          point_to,
                          tcodes=tcodes,
                          early_border=point_from.localize(loc=datetime.combine(when, time(0, 0))),
                          late_border=point_from.localize(loc=datetime.combine(when + timedelta(days=1), time(4, 0))),
                          user_settlement=request.client_city)

    for segment in segments:
        segment.display_info['tariffs_info'] = cook_tariffs(segment.info, segment.t_type.code)


def add_station_station_tariffs(segments):
    segments = itertools.ifilterfalse(operator.attrgetter('gone'), segments)

    segments = filter(lambda s: s.t_type.code in ('suburban', 'train'), segments)

    def segment_stations(segment):
        return segment.station_from.id, segment.station_to.id

    segments = sorted(segments, key=segment_stations)

    for stations, segments in itertools.groupby(segments, segment_stations):
        segments = list(segments)

        tariffs = get_tariffs_for_segments(segments)

        for segment in segments:
            tariff_type = segment.thread.tariff_type

            if tariff_type:
                code = segment.thread.tariff_type.code
            else:
                if segment.thread.is_express or segment.thread.is_aeroexpress:
                    code = 'express'
                else:
                    code = 'etrain'

            if code in tariffs:
                segment.display_info.set_tariff(tariffs[code].tariff)


def get_tariffs_for_segments(segments):
    # Пытаемся найти первую нитку без изменений
    segment = next(itertools.ifilterfalse(lambda s: s.thread.type.code == 'change', segments),
                   segments[0])

    currency_from = getattr(segment.station_from.country, 'currency', None)

    return get_tariffs(segment, currency=getattr(currency_from, 'code', None))


def get_tariff_qs_for_segment(segment):
    """Условия для выборки тарифа"""

    return AeroexTariff.objects.filter(
        Q(station_from=segment.station_from, station_to=segment.station_to) |
        Q(station_to=segment.station_from, station_from=segment.station_to, reverse=True)
    ).order_by('-precalc')  # Чтобы ручные забивали предрасчитанные


def get_tariffs(segment, currency=None):
    tariffs_qs = get_tariff_qs_for_segment(segment)

    tariffs = {}

    for tariff in tariffs_qs:
        tariff.tariff = Price(tariff.tariff, currency or tariff.currency)

        tariffs[tariff.type.code] = tariff

    return tariffs
