# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

import requests
from django.conf import settings
from marshmallow import ValidationError

from travel.rasp.library.python.common23.models.core.geo.station import Station
from travel.rasp.library.python.common23.utils.geo import great_circle_distance_km

from travel.rasp.blablacar.blablacar.clients.base_client import BaseClient, ConnectionError
from travel.rasp.blablacar.blablacar.clients.blablacar_v3.serialization import (
    BlablacarV3ResponseSchema, BlablacarV3ResponseParseError
)
from travel.rasp.blablacar.blablacar.clients.blablacar_v3.tariff import create_response
from travel.rasp.blablacar.blablacar.clients.const import (
    BLABLACAR_API_KEY, BLABLACAR_PROCESSING_ERRORS, BLABLACAR_SPECIAL_ERRORS_BY_STATUS_CODE
)
from travel.rasp.blablacar.blablacar.const import BLABLACAR_ALL_DAYS_DATE, BLABLACAR_NATIONAL_SETTINGS
from travel.rasp.blablacar.blablacar.service.response_prototype import (
    ResponsePrototype, build_error_response_and_cache_timeout
)

log = logging.getLogger(__name__)

BLABLACAR_URL = 'https://public-api.blablacar.com/api/v3/trips'


def get_search_point_coords(point):
    if isinstance(point, Station) and point.settlement:
        search_point = point.settlement
    else:
        search_point = point

    return '{},{}'.format(search_point.latitude, search_point.longitude)


class BlablacarV3Client(BaseClient):
    def __init__(
        self, url=BLABLACAR_URL, api_timeout=settings.BLABLACAR_API_TIMEOUT, verify_ssl=settings.BLABLACAR_VERIFY_SSL
    ):
        super(BlablacarV3Client, self).__init__(log, url, api_timeout, verify_ssl)

    def _make_params(self, query):
        national_settings = BLABLACAR_NATIONAL_SETTINGS.get(
            query.national_version, BLABLACAR_NATIONAL_SETTINGS['ru']
        )
        params = {
            'from_coordinate': get_search_point_coords(query.point_from),
            'to_coordinate': get_search_point_coords(query.point_to),
            'locale': national_settings['locale'],
            'currency': national_settings['blablacar_currency'],
            'key': BLABLACAR_API_KEY,
            'sort': 'price:asc',
            'count': 1,
            'requested_seats': 1
        }
        if query.date != BLABLACAR_ALL_DAYS_DATE:
            date_string = query.date.strftime('%Y-%m-%dT%H:%M:%S')
            params['start_date_local'] = date_string
            params['end_date_local'] = date_string
        return params

    def make_response_and_cache_timeout(self, query):
        try:
            raw_response = self._get_raw_response(query)
        except ConnectionError:
            return build_error_response_and_cache_timeout(BLABLACAR_PROCESSING_ERRORS['communication_error'])

        if raw_response.status_code in BLABLACAR_SPECIAL_ERRORS_BY_STATUS_CODE:
            log.info('Специальный случай для кода ответа %s от blablacar', raw_response.status_code)
            return build_error_response_and_cache_timeout(BLABLACAR_SPECIAL_ERRORS_BY_STATUS_CODE[raw_response.status_code])

        try:
            raw_response.raise_for_status()
        except requests.HTTPError:
            log.exception('Неуспешный ответ блаблакара')
            return build_error_response_and_cache_timeout(BLABLACAR_PROCESSING_ERRORS['bad_response'])

        try:
            parsed_response, parse_errors = BlablacarV3ResponseSchema().loads_old(raw_response.content)
            if parse_errors:
                raise ValidationError(parse_errors)
        except (BlablacarV3ResponseParseError, ValueError, ValidationError) as err:
            log.exception('Не удалось разобрать ответ от блаблакар: %s', err)
            return build_error_response_and_cache_timeout(BLABLACAR_PROCESSING_ERRORS['unparseable_response'])

        created_response, error = create_response(parsed_response, query)

        if error:
            return build_error_response_and_cache_timeout(BLABLACAR_PROCESSING_ERRORS[error])

        response = ResponsePrototype({
            'querying': False,
            'banned': False,
            'distance': great_circle_distance_km(query.point_from, query.point_to),
            'tariff': created_response.to_response(),
        })

        return response, settings.BLABLACAR_SUCCESS_CACHE_TIMEOUT
