# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

import requests
from django.conf import settings
from marshmallow import ValidationError

from travel.rasp.library.python.common23.utils.geo import great_circle_distance_km

from travel.rasp.library.python.common23.models.core.geo.station import Station
from travel.rasp.blablacar.blablacar.clients.const import EDEM_RF_PROCESSING_ERRORS
from travel.rasp.blablacar.blablacar.clients.base_client import BaseClient, ConnectionError
from travel.rasp.blablacar.blablacar.clients.edem_rf.serialization import EdemRfResponseSchema
from travel.rasp.blablacar.blablacar.clients.edem_rf.tariff import create_response
from travel.rasp.blablacar.blablacar.service.response_prototype import (
    ResponsePrototype, build_error_response_and_cache_timeout
)


log = logging.getLogger(__name__)

EDEM_RF_URL = 'https://xn--d1abb2a.xn--p1ai/routes/export'


def get_search_point_title(point):
    search_point = point.settlement if isinstance(point, Station) and point.settlement else point
    return search_point.L_title()


class EdemRfClient(BaseClient):
    def __init__(
        self,
        url=EDEM_RF_URL,
        api_timeout=settings.EDEM_RF_API_TIMEOUT,
        verify_ssl=settings.EDEM_RF_VERIFY_SSL
    ):
        super(EdemRfClient, self).__init__(log, url, api_timeout, verify_ssl)

    def _make_params(self, query):
        return {
            'fromCityName': get_search_point_title(query.point_from),
            'toCityName': get_search_point_title(query.point_to),
            'createdDate': query.date.strftime('%Y-%m-%d')
        }

    def make_response_and_cache_timeout(self, query):
        try:
            raw_response = self._get_raw_response(query)
        except ConnectionError:
            return build_error_response_and_cache_timeout(EDEM_RF_PROCESSING_ERRORS['communication_error'])

        try:
            raw_response.raise_for_status()
        except requests.HTTPError as ex:
            log.exception('Неуспешный ответ едем.рф. Код ответа: {}'.format(ex.response.status_code))
            return build_error_response_and_cache_timeout(EDEM_RF_PROCESSING_ERRORS['bad_response'])

        try:
            parsed_response, parse_errors = EdemRfResponseSchema().loads_old(raw_response.content)
            if parse_errors:
                raise ValidationError(parse_errors)
        except (ValidationError, ValueError) as ex:
            log.exception('Не удалось разобрать ответ от едем.рф: %s', ex)
            return build_error_response_and_cache_timeout(EDEM_RF_PROCESSING_ERRORS['unparseable_response'])

        response = ResponsePrototype({
            'querying': False,
            'banned': False,
            'distance': great_circle_distance_km(query.point_from, query.point_to),
            'tariff': create_response(parsed_response, query).to_response(),
        })
        return response, settings.EDEM_RF_SUCCESS_CACHE_TIMEOUT
