# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.conf import settings
from django.utils.functional import cached_property
from marshmallow import validates_schema, post_load, ValidationError, fields

from travel.rasp.library.python.common23.models.core.geo.country import Country
from travel.rasp.library.python.common23.models.core.geo.point import Point
from travel.rasp.library.python.common23.models.serialization.compat import Schema
from travel.rasp.blablacar.blablacar.const import BLABLACAR_ALL_DAYS_DATE, CarProvider

from travel.rasp.library.python.common23.utils.namedtuple import namedtuple_with_defaults


class PointField(fields.Field):
    def _deserialize(self, value, attr, data, **kwargs):
        try:
            return Point.get_by_key(value)
        except Exception:
            raise ValidationError('Wrong value "{}" for PointField'.format(value))

    def _serialize(self, value, attr, obj, **kwargs):
        return value.point_key


class BlablacarQuery(object):
    def __init__(self, point_from, point_to, minimal_distance=None, date=BLABLACAR_ALL_DAYS_DATE,
                 national_version='ru', is_touch=False, provider=CarProvider.BLABLACAR):
        self.date = date
        self.point_from = point_from
        self.point_to = point_to
        self.minimal_distance = minimal_distance or settings.BLABLACAR_MIN_DISTANCE_KM
        self.national_version = national_version
        self.is_touch = is_touch
        self.provider = provider

    @cached_property
    def key(self):
        fields_key = '{date}&{point_from_key}&{point_to_key}&{national_version}&{is_touch}&{provider}'.format(
            date=self.date,
            point_from_key=self.point_from.point_key,
            point_to_key=self.point_to.point_key,
            national_version=self.national_version,
            is_touch=self.is_touch,
            provider=self.provider
        )
        return (settings.BLABLACAR_CACHEROOT + fields_key).replace(' ', ':')

    def __eq__(self, other):
        return self.key == other.key


class BlablacarQuerySchema(Schema):
    date = fields.Date(data_key='date')
    point_from = PointField(data_key='pointFrom', required=True)
    point_to = PointField(data_key='pointTo', required=True)
    national_version = fields.String()
    is_touch = fields.Boolean(data_key='isTouch')
    minimal_distance = fields.Integer(data_key='minimalDistance')
    provider = fields.String()

    @validates_schema(skip_on_field_errors=False)
    def validate_request(self, data, **kwargs):
        schema_errors = {}

        if data.get('provider') is not None:
            if data['provider'] in CarProvider.BLABLACAR_PROVIDERS:
                data['provider'] = CarProvider.BLABLACAR
            elif data['provider'] not in CarProvider.ALL:
                schema_errors['provider'] = 'unknown_provider'

        for field in ('point_from', 'point_to'):
            if data.get(field) is None:
                schema_errors[field] = 'no_such_point'
            elif isinstance(data[field], Country):
                schema_errors[field] = 'country_point'

        if not schema_errors:
            if data['point_from'] == data['point_to']:
                schema_errors['same_points'] = 'same_points'

        if schema_errors:
            raise ValidationError(schema_errors)

    @post_load
    def make_blablacar_query(self, data, **kwargs):
        if 'minimal_distance' not in data:
            data['minimal_distance'] = settings.BLABLACAR_MIN_DISTANCE_KM
        return BlablacarQuery(**data)


SettlementQuery = namedtuple_with_defaults('SettlementQuery', [
    'id', 'national_version', 'language', 'provider'
], defaults={'national_version': 'ru', 'language': 'ru', 'provider': CarProvider.BLABLACAR})


class SettlementQuerySchema(Schema):
    id = fields.Integer(required=True)
    national_version = fields.String(data_key='nationalVersion', missing='ru')
    language = fields.String(missing='ru')
    provider = fields.String()

    @validates_schema(skip_on_field_errors=False)
    def validate_request(self, data, **kwargs):
        if data.get('provider') is not None and data['provider'] not in CarProvider.ALL:
            raise ValidationError({'provider': 'unknown_provider'})

    @post_load
    def make_context(self, data, **kwargs):
        return SettlementQuery(**data)


class SettlementResponseSchema(Schema):
    url = fields.String()
    banned = fields.Boolean()
    errors = fields.Dict()
