# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from future import standard_library
standard_library.install_aliases()
from django.utils.encoding import force_text
from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response
from ylog.context import log_context
from urllib.parse import quote

from travel.rasp.blablacar.blablacar.const import BLABLACAR_NATIONAL_SETTINGS, CarProvider
from travel.rasp.blablacar.blablacar.data_layer import get_visible_settlement
from travel.rasp.blablacar.blablacar.serialization import (
    BlablacarQuerySchema, SettlementQuerySchema, SettlementResponseSchema
)
from travel.rasp.blablacar.blablacar.service.distance import check_blablacar_distance, BlablacarDistanceError
from travel.rasp.blablacar.blablacar.service.query_and_cache import BANNED_RESPONSE, process_find_routes_query
from travel.rasp.blablacar.blablacar.service.restriction import is_direction_banned, is_settlement_banned


@api_view(['GET'])
def blablacar(request, poll=False):
    # Пример запроса к ручке: ?date=2019-11-08&pointFrom=c213&pointTo=c16&national_version=ru
    query, errors = BlablacarQuerySchema().load_old(request.GET)

    if errors:
        return Response({'errors': errors}, status=status.HTTP_400_BAD_REQUEST)

    if query.provider in CarProvider.BLABLACAR_PROVIDERS:
        if is_direction_banned(query.point_from, query.point_to):
            return BANNED_RESPONSE.build_response()

        try:
            check_blablacar_distance(query.point_from, query.point_to, query.minimal_distance)
        except BlablacarDistanceError as e:
            return Response({'errors': {'distance': force_text(e)}}, status=status.HTTP_428_PRECONDITION_REQUIRED)

    with log_context(direction='{}-{}'.format(query.point_from.point_key, query.point_to.point_key)):
        find_router_response = process_find_routes_query(query, poll)
        return find_router_response.build_response()


@api_view(['GET'])
def settlement(request):
    """
    [RASPFRONT-7674] Ручка для формирования ссылки на блаблакар для городов
    Пример запроса к ручке: /settlement/?id=213&national_version=ru&language=ru
    """

    settlement_query, errors = SettlementQuerySchema().load_old(request.GET)

    if errors:
        return Response({'errors': errors}, status=status.HTTP_400_BAD_REQUEST)

    settlement = get_visible_settlement(settlement_query.id)

    if settlement is None:
        return Response({'errors': {'settlement': 'not found'}}, status=status.HTTP_400_BAD_REQUEST)

    if settlement_query.provider in CarProvider.BLABLACAR_PROVIDERS and is_settlement_banned(settlement):
        return Response(SettlementResponseSchema().dump_old({'url': '', 'banned': True}).data)

    settlement_title = settlement.L_title_with_full_geography(
        settlement_query.national_version, settlement_query.language
    )
    national_settings = BLABLACAR_NATIONAL_SETTINGS.get(
        settlement_query.national_version, BLABLACAR_NATIONAL_SETTINGS['ru']
    )
    url = '{base_url}fn={title}'.format(
        base_url=national_settings['redirect_base_url'],
        title=quote(settlement_title.encode('utf-8'))
    )

    return Response(SettlementResponseSchema().dump_old({'url': url, 'banned': False}).data)
