# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import os.path
from rasp_vault.api import get_secret

from travel.rasp.library.python.common23.settings import *  # noqa
from travel.rasp.library.python.common23.settings import PROJECT_PATH, COMMON_CONTRIB_APPS, COMMON_INSTALLED_APPS_BASE, TEMPLATES
from travel.rasp.library.python.common23.settings.configuration import Configuration, BaseConfigurator
from travel.rasp.library.python.common23.settings.utils import configure_raven, get_app_version
from travel.rasp.library.python.common23.settings.configuration import BaseRedisCacheConfigurator

from travel.rasp.library.python.common23.logging.configure import configure_default_logging


STATIC_URL = '/static/'
STATIC_ROOT = os.path.join(PROJECT_PATH, 'static')

INSTALLED_APPS = COMMON_CONTRIB_APPS + [
    'django_opentracing',
    'django.contrib.staticfiles',
    'django.contrib.sites',
    'rest_framework',
    'raven.contrib.django.raven_compat',
] + COMMON_INSTALLED_APPS_BASE + [
    'travel.rasp.blablacar.blablacar'
]

MIDDLEWARE_CLASSES = [
    'django_opentracing.OpenTracingMiddleware',
    'travel.library.python.tracing.django.TracingMiddleware',
    'travel.rasp.library.python.common23.middleware.extract_context.ExtractRequestMiddleware',
    'travel.rasp.library.python.common23.middleware.ping.PingResponder',
    'travel.rasp.library.python.common23.middleware.syncdb.SyncDB',
]

TEMPLATES.append(
    {
        'BACKEND': 'library.python.django.template.backends.arcadia.ArcadiaTemplates',
        'OPTIONS': {
            'debug': False,
            'loaders': [
                'library.python.django.template.loaders.resource.Loader',
                'library.python.django.template.loaders.app_resource.Loader',
            ],
            'context_processors': [
                'django.template.context_processors.debug',
                'django.template.context_processors.request',
                'django.contrib.auth.context_processors.auth',
                'django.contrib.messages.context_processors.messages',
            ],
        },
    }
)


def context_extractor(request):
    return {
        'user_rid': request.GET.get('_rid', ''),
        'service_rid': request.META.get('HTTP_X_REQUEST_ID', '')
    }


PYLOGCTX_REQUEST_EXTRACTOR = context_extractor

OPENTRACING_TRACE_ALL = True
OPENTRACING_TRACED_ATTRIBUTES = ['path', 'method']

ROOT_URLCONF = 'travel.rasp.blablacar.urls'
ALLOWED_HOSTS = ['.yandex.ru', '.yandex.net', '.yandex-team.ru', '.yandex-team.net']


PKG_VERSION = get_app_version()


class BlablacarRedisConfigurator(BaseRedisCacheConfigurator):
    CLUSTER_ID_SECRET = 'REDIS_BLABLACAR_CLUSTER_ID'
    CLUSTER_PASSWORD_SECRET = 'REDIS_BLABLACAR_PASSWORD'
    CLUSTER_NAME_SECRET = 'REDIS_BLABLACAR_NAME'
    CACHE_ALIAS = 'blablacar'
    FALLBACK_HOSTS = {
        'testing': [
            'man-8evba5pkdk4z36f4.db.yandex.net',
            'sas-vx4exvv2tew4wodp.db.yandex.net',
            'vla-nn2a43lrlqd6llwu.db.yandex.net'
        ],
        'production': [
            'man-as2yqzxpvj40ssab.db.yandex.net',
            'sas-qb02735mm6gpcfxo.db.yandex.net',
            'vla-13xm8bpafjexwgc0.db.yandex.net'
        ]
    }

    def apply_base(self, settings):
        settings['BLABLACAR_CACHE_CLUSTER_ALIAS'] = self.CACHE_ALIAS

    def apply_testing(self, settings):
        self.configure_redis(settings, get_secret('rasp-blablacar-testing'))

    def apply_production(self, settings):
        self.configure_redis(settings, get_secret('rasp-blablacar-production'))


class BlablacarConfigurator(BaseConfigurator):
    def apply_base(self, settings):
        settings['LOG_PATH'] = settings.get('LOG_PATH', '/var/log/app')
        settings['BLABLACAR_CACHEROOT'] = '/blablacar/{}/'.format(settings['APPLIED_CONFIG'])
        self.setup_logging(settings)

    def apply_testing(self, settings):
        configure_raven(settings, 'blablacar')

    def apply_production(self, settings):
        configure_raven(settings, 'blablacar')

    def apply_development(self, settings):
        settings['LOG_PATH'] = os.path.join(PROJECT_PATH, 'log')
        self.setup_logging(settings)

    @classmethod
    def setup_logging(cls, settings, base_log_level=None):
        settings['LOGGING'] = configure_default_logging(settings, base_log_level)


Configuration.configurator_classes += [BlablacarRedisConfigurator, BlablacarConfigurator]
if os.getenv('DJANGO_SETTINGS_MODULE') == 'settings':
    Configuration().apply(globals())
