import logging

import click
from flask.cli import FlaskGroup
from flask.logging import default_handler
from flask_migrate import Migrate
from flask_migrate.cli import db as cliDb

from travel.library.python.resource import extract_resources
from travel.rasp.bus.db.models.shared import Base
from travel.rasp.bus.settings import Settings
from travel.rasp.bus.admin.app import create_bus_admin
from travel.rasp.bus.admin.utils.logging import GunicornLogger


@click.command()
@click.option('--bind', '-b', type=str, default='[::]:5000', help='Bind', show_default=True)
@click.option('--workers', '-w', type=int, default=4, help='Number of workers to spawn', show_default=True)
@click.option('--timeout', '-t', type=int, default=30, help='Worker timeout', show_default=True)
def gunicorn(bind, workers, timeout):
    """Start the Server with Gunicorn"""
    from gunicorn.app.base import BaseApplication

    class FlaskApplication(BaseApplication):
        def __init__(self):
            self.options = {
                'bind': bind,
                'workers': workers,
                'logger_class': GunicornLogger,
                'timeout': timeout,
            }
            super().__init__()

        def load_config(self):
            config = {
                key: value
                for key, value in self.options.items()
                if key in self.cfg.settings and value is not None
            }
            for key, value in config.items():
                self.cfg.set(key.lower(), value)

        def load(self):
            return create_bus_admin()

    application = FlaskApplication()

    root_logger = logging.getLogger()
    root_logger.setLevel(Settings.LOGGING_LEVEL)
    root_logger.addHandler(default_handler)

    for silenced_logger in ('urllib3', 'zeep', 'kikimr', 'rasp.bus.admin.utils.logbroker_client'):
        logging.getLogger(silenced_logger).setLevel(logging.INFO)

    return application.run()


def main():

    extract_resources('travel/rasp/bus/admin/')

    def create_db_app():
        bus_admin = create_bus_admin()
        Migrate(bus_admin, Base)
        return bus_admin

    cli = FlaskGroup(create_app=create_db_app)
    cli.add_command(cliDb)
    cli.add_command(gunicorn)
    cli()


if __name__ == '__main__':
    main()
