import yenv
from flask import Flask, render_template, redirect, url_for
from flask_admin.base import Admin
from flask_passport import Authenticator, PassportClient, login_exempt_views
from flask_principal import identity_loaded, RoleNeed
from flask_sqlalchemy import SQLAlchemy
from jinja2 import FileSystemLoader
from travel.rasp.bus.db import session_scope
from travel.rasp.bus.db.models.admin_user import AdminUser
from travel.rasp.bus.db.models.carrier import Carrier
from travel.rasp.bus.db.models.carrier_matching import CarrierMatching
from travel.rasp.bus.db.models.matching import PointMatching, PointMatchingLog
from travel.rasp.bus.db.models.register_type import RegisterType
from travel.rasp.bus.db.models.shared import metadata
from travel.rasp.bus.db.models.supplier import Supplier
from travel.rasp.bus.settings import Settings

from travel.rasp.bus.admin.app.idm import IdmApi
from travel.rasp.bus.admin.app.views.admin_user import AdminUserView
from travel.rasp.bus.admin.app.views.carrier import CarrierView
from travel.rasp.bus.admin.app.views.carrier_matching import CarrierMatchingView
from travel.rasp.bus.admin.app.views.point_matching import PointMatchingView
from travel.rasp.bus.admin.app.views.point_matching_logs import PointMatchingLogsView
from travel.rasp.bus.admin.app.views.index import AdminIndexView
from travel.rasp.bus.admin.app.views.carrier_search import CarrierSearchView
from travel.rasp.bus.admin.app.views.register_type import RegisterTypeView
from travel.rasp.bus.admin.app.views.supplier import SupplierView


def create_bus_admin():
    bus_admin = Flask(__name__)
    bus_admin.register_blueprint(IdmApi(), url_prefix='/idm')

    bus_admin.jinja_loader = FileSystemLoader('app/templates/')
    bus_admin.config.from_object(Settings)
    db = SQLAlchemy(metadata=metadata)
    db.init_app(bus_admin)

    admin = Admin(bus_admin, template_mode='bootstrap3',
                  base_template='custom_master.html',
                  index_view=AdminIndexView())
    admin.add_views(
        AdminUserView(AdminUser, db.session),
        CarrierView(Carrier, db.session),
        CarrierMatchingView(CarrierMatching, db.session),
        PointMatchingView(PointMatching, db.session),
        PointMatchingLogsView(PointMatchingLog, db.session),
        RegisterTypeView(RegisterType, db.session),
        SupplierView(Supplier, db.session),
        CarrierSearchView(name='Find Carrier')
    )

    @bus_admin.context_processor
    def injects_yenv():
        return {'yenv_type': yenv.type}

    @bus_admin.errorhandler(403)
    def forbidden(_):
        return render_template('errors/403.html'), 403

    @bus_admin.after_request
    def remove_session(response):
        db.session.remove()
        return response

    @bus_admin.route('/')
    def home_page():
        return redirect(url_for('admin.index'))

    @bus_admin.route('/ping')
    def ping():
        return 'pong'

    login_exempt_views.update({
        'travel.rasp.bus.admin.app.ping',
        'travel.rasp.bus.admin.app.idm.info',
        'travel.rasp.bus.admin.app.idm.remove_role',
        'travel.rasp.bus.admin.app.idm.add_role',
        'travel.rasp.bus.admin.app.idm.get_all_roles',
    })

    Authenticator(
        bus_admin,
        enable=yenv.type in ('production', 'testing'),
        passport=PassportClient(
            tvm_id=Settings.Admin.TVM_CLIENT_ID,
            tvm_secret=Settings.Admin.TVM_SECRET,
        )
    )

    @identity_loaded.connect_via(bus_admin)
    def on_identity_loaded(_, identity):
        with session_scope() as session:
            login = identity.id
            if not login:
                return
            user = session.query(AdminUser).get(login)
            if not user:
                return
            identity.user = user
            for role in user.roles:
                if role.is_active:
                    identity.provides.add(RoleNeed(role.name))

    return bus_admin
