import json
import logging
from functools import wraps

from flask import Blueprint, jsonify, request
from sqlalchemy import func
from sqlalchemy.exc import DBAPIError
from travel.rasp.bus.db import session_scope
from travel.rasp.bus.db.models.admin_user import AdminUser, AdminUserRole
from travel.rasp.bus.library.tvm import TvmChecker
from travel.rasp.bus.roles import ROLES
from travel.rasp.bus.settings import Settings
from ticket_parser2.api.v1 import BlackboxClientId
from werkzeug.exceptions import Forbidden


class IdmApi(Blueprint):

    def __init__(self):
        super().__init__('idm', __name__)
        if Settings.IDM.TVM_CHECK_ENABLED:
            self._tvm = TvmChecker(
                client_id=Settings.Admin.TVM_CLIENT_ID,
                blackbox_client=BlackboxClientId.Prod,
                allowed_clients=(Settings.IDM.TVM_CLIENT_ID,),
                secret=Settings.Admin.TVM_SECRET
            )
        self.errorhandler(DBAPIError)(self.handle_operational_error)
        self.errorhandler(Exception)(self.handle_exception)
        self.route('/info/')(self.info)
        self.route('/add-role/', methods=['POST'])(self.add_role)
        self.route('/remove-role/', methods=['POST'])(self.remove_role)
        self.route('/get-all-roles/')(self.get_all_roles)

    def route(self, rule, **options):
        def decorator(f):
            @wraps(f)
            def wrapped_f():
                if not self.check_request_ticket():
                    raise Forbidden('TVM check fails')
                return f()
            super(IdmApi, self).route(rule, **options)(wrapped_f)
            return wrapped_f
        return decorator

    def check_request_ticket(self):
        return not Settings.IDM.TVM_CHECK_ENABLED or self._tvm.check_request_ticket()

    @staticmethod
    def handle_operational_error(err):
        logging.error('Database error: %r', err)
        return jsonify({
            'code': 2,
            'error': 'DBAPIError'
        })

    @staticmethod
    def handle_exception(err):
        logging.error('Unhandled exception: %r', err)
        return jsonify({
            'code': 3,
            'fatal': 'Unhandled exception'
        })

    @staticmethod
    def info():
        return jsonify({
            'code': 0,
            'roles': {
                'slug': 'role',
                'name': 'роль',
                'values': ROLES.asdict()
            }
        })

    @staticmethod
    def add_role():
        login = request.form['login']
        role = json.loads(request.form['role'])['role']

        with session_scope() as session:
            user_role = session.query(AdminUserRole).get((login, role))

            if user_role and user_role.is_active:
                return jsonify({
                    'code': 1,
                    'warning': 'AdminUser already has that role'
                })

            if user_role:
                user_role.is_active = True
            else:
                user = session.query(AdminUser).get(login) or AdminUser(login)
                session.add(user)
                user_role = AdminUserRole(user, role)
                session.add(user_role)
            return jsonify({
                'code': 0
            })

    @staticmethod
    def remove_role():
        login = request.form['login']
        role = json.loads(request.form['role'])['role']

        with session_scope() as session:
            user_role = session.query(AdminUserRole).get((login, role))

            if not user_role or not user_role.is_active:
                return jsonify({
                    'code': 1,
                    'warning': 'AdminUser has not that role'
                })

            user_role.is_active = False

        return jsonify({
            'code': 0
        })

    @staticmethod
    def get_all_roles():
        with session_scope() as session:
            user_roles = session.query(AdminUserRole).filter(
                AdminUserRole.is_active.is_(True)
            ).with_entities(
                AdminUserRole.yandex_login,
                func.array_agg(
                    func.json_build_object(
                        'role',
                        AdminUserRole.name
                    )
                )
            ).group_by(
                AdminUserRole.yandex_login
            ).order_by(
                AdminUserRole.yandex_login
            ).all()

        resp = {
            'code': 0,
            'users': [
                {
                    'login': yandex_login,
                    'roles': roles
                }
                for yandex_login, roles
                in user_roles
            ]
        }
        return jsonify(resp)
