import wtforms
from flask import flash, redirect, request
from flask_admin import BaseView, expose, helpers
from flask_admin.form.widgets import RenderTemplateWidget
from travel.rasp.bus.db import session_scope
from travel.rasp.bus.db.models.supplier import Supplier

from travel.rasp.bus.admin.utils.carriers.cache import dump_carriers, load_carriers
from travel.rasp.bus.library.carrier import CarrierType
from travel.rasp.bus.library.carrier.matching import CarrierMatchingError, create_carrier_and_matching
from travel.rasp.bus.spark_api.spark_search import CarrierSearchError, find_carriers
from travel.rasp.bus.admin.app.views.principal import PrincipalViewMixin


class CarrierSearchForm(wtforms.Form):
    carrier_type = wtforms.RadioField(
        choices=((CarrierType.ENTREPRENEUR, 'ИП'), (CarrierType.COMPANY, 'Компания'),),
        coerce=lambda v: CarrierType(int(v)) if not isinstance(v, CarrierType) else v,
        default=CarrierType.ENTREPRENEUR
    )
    code = wtforms.StringField(
        default='',
        validators=(wtforms.validators.DataRequired(),)
    )


class CarrierMatchForm(wtforms.Form):
    carrier_type = wtforms.IntegerField(
        validators=(wtforms.validators.InputRequired(),),
        widget=wtforms.widgets.HiddenInput(),
    )
    carrier_cache_code = wtforms.StringField(
        validators=(wtforms.validators.DataRequired(),),
        widget=wtforms.widgets.HiddenInput(),
    )
    carrier_cache_index = wtforms.IntegerField(
        validators=(wtforms.validators.InputRequired(),)
    )
    carrier_code = wtforms.StringField(
        validators=(wtforms.validators.DataRequired(),)
    )
    suppliers = wtforms.SelectMultipleField(
        coerce=int,
        validators=(wtforms.validators.InputRequired(),),
        widget=RenderTemplateWidget('checkboxes_field.html')
    )

    def __init__(self, *args, **kwargs):
        super(CarrierMatchForm, self).__init__(*args, **kwargs)
        self.set_supplier_choices()

    def set_supplier_choices(self):
        with session_scope() as session:
            self.suppliers.choices = [(s.id, s.code) for s in session.query(Supplier).order_by(Supplier.code)]


class CarrierSearchView(PrincipalViewMixin, BaseView):
    @expose('/')
    def index_view(self):
        search_form = CarrierSearchForm(request.args)
        carrier_type = carriers = match_form = None
        if request.args:
            if search_form.validate():
                carrier_type, carrier_name_or_code = search_form.carrier_type.data, search_form.code.data.strip()
                is_name_search = not carrier_name_or_code.isdecimal()

                carriers = load_carriers(carrier_type, carrier_name_or_code)
                if carriers is None:
                    try:
                        carriers = find_carriers(carrier_type, carrier_name_or_code, is_name_search=is_name_search)
                    except CarrierSearchError as exc:
                        flash(str(exc), 'error')
                    else:
                        dump_carriers(carrier_type, carrier_name_or_code, carriers)

                match_form = CarrierMatchForm()
                match_form.carrier_type.data = carrier_type.value

                if is_name_search:
                    match_form.carrier_cache_code.data = match_form.carrier_code.data = carrier_name_or_code
            else:
                helpers.flash_errors(search_form, message='Search failed. %(error)s')

        return self.render(
            'carrier_search.html',
            carrier_type=carrier_type,
            carriers=carriers,
            match_form=match_form,
            search_form=search_form,
        )

    @expose('/match/', methods=('POST',))
    def match_view(self):
        match_form = CarrierMatchForm(request.form)

        if match_form.validate():
            carrier_type_id, carrier_cache_code, carrier_cache_index, carrier_code, supplier_ids = (
                match_form.carrier_type.data,
                match_form.carrier_cache_code.data,
                match_form.carrier_cache_index.data,
                match_form.carrier_code.data,
                match_form.suppliers.data
            )
            carrier_type = CarrierType(carrier_type_id)

            carriers = load_carriers(carrier_type, carrier_cache_code)
            if carriers is None:
                flash('Carrier search cache is empty. Please run search again.', 'error')
            else:
                carrier = carriers[carrier_cache_index]
                try:
                    carrier_id, matching_ids = create_carrier_and_matching(
                        carrier_type, carrier_code, carrier, supplier_ids
                    )
                except CarrierMatchingError as exc:
                    flash(str(exc), 'error')
                else:
                    flash(f'Carrier matching created (carrier_id: {carrier_id}, matching_ids: {matching_ids})')
        else:
            helpers.flash_errors(match_form, message='Match failed. %(error)s')

        return redirect(self.get_url('.index_view'))
