import json
import logging
from collections import defaultdict
from re import search as re_search

import requests
from flask import jsonify, request, g, redirect, flash
from flask_admin import expose
from flask_admin.babel import gettext
from flask_admin.contrib.sqla import ModelView
from flask_admin.contrib.sqla.filters import BooleanEqualFilter, FilterEqual, FilterEmpty, FilterLike
from flask_admin.helpers import get_url
from flask_admin.model.ajax import AjaxModelLoader
from flask_admin.model.widgets import XEditableWidget
from sqlalchemy import func
from travel.rasp.bus.db import session_scope
from travel.rasp.bus.db.models.matching import PointMatching, PointType as MatchingPointType, MatchingChange
from travel.rasp.bus.db.models.supplier import Supplier
from travel.rasp.bus.roles import ROLES
from travel.rasp.bus.settings import Settings
from werkzeug import exceptions
from wtforms import fields

from travel.rasp.bus.admin.app.views.principal import PrincipalViewMixin
from travel.rasp.bus.admin.utils.points.raspadmin import PointType, RaspAdminPointAdapter
from travel.rasp.bus.admin.utils.points.suggest import SuggestClient


class SuggestsLoader(AjaxModelLoader):

    def __init__(self, options):
        super().__init__('suggest', options)

    def format(self, pk):
        if pk:
            rasp_point = SuggestClient.get_by_point_key(pk)
            return pk, pk + ': ' + (rasp_point or {}).get('full_title', 'not found')
        return None

    def get_one(self, pk):
        return pk


class SuggestWidget(XEditableWidget):

    def get_kwargs(self, field, kwargs):
        if field.name == 'point_key':
            kwargs['data-role'] = 'x-editable-select2-ajax'
            kwargs['data-lookup-url'] = get_url('.ajax_lookup', name=field.name)
            kwargs['data-allow-blank'] = '1'

            if field.data:
                kwargs['value-pointkey'] = field.data
                kwargs['data-value'] = field.data

            placeholder = gettext('Start typing')
            kwargs.setdefault('data-placeholder', placeholder)
        else:
            kwargs = super().get_kwargs(field, kwargs)
        return kwargs


def _get_supplier_filter_options():
    if not request:
        return
    with session_scope() as session:
        return [(s.id, s.name) for s in session.query(Supplier)]


class NullableStringField(fields.StringField):
    def process_formdata(self, valuelist):
        if valuelist:
            self.data = valuelist[0] if valuelist[0] else None


class PointMatchingView(PrincipalViewMixin, ModelView):
    can_create = False
    can_delete = False
    can_edit = False
    can_view_details = True
    page_size = 50
    can_set_page_size = True
    edit_modal = False
    list_template = 'point_matching_list.html'
    details_template = "point_matching_details.html"
    accessible_for = [ROLES.Admin, ROLES.PointMatching]

    column_list = [
        'title',
        'description',
        'type',
        'supplier',
        'point_key',
        'outdated',
        'disabled',
        'in_segments',
    ]

    column_filters = [
        FilterEqual(column=PointMatching.supplier_id, name='Supplier', options=_get_supplier_filter_options),
        FilterEqual(column=PointMatching.supplier_point_id, name='Supplier Point Id'),
        FilterEqual(column=PointMatching.type, name='Supplier Point Type', options=tuple(
            (point_type.name, point_type.name) for point_type in MatchingPointType
        )),
        FilterEqual(column=PointMatching.point_key, name='Point Key'),
        FilterEmpty(column=PointMatching.point_key, name='Point Key'),
        FilterLike(column=PointMatching.point_key, name='Point Key', options=tuple(
            (f'^{point_type.prefix}', point_type.name) for point_type in PointType
        )),
        BooleanEqualFilter(column=PointMatching.outdated, name='Outdated'),
        BooleanEqualFilter(column=PointMatching.disabled, name='Disabled'),
        BooleanEqualFilter(column=PointMatching.in_segments, name='In segments'),
    ]

    column_editable_list = [
        'point_key',
        'disabled',
    ]

    column_searchable_list = [
        'title',
        'description',
        'supplier_point_id',
        'point_key',
    ]

    column_default_sort = ('point_key', True)

    column_sortable_list = [
        'title',
        'type',
        'point_key',
        'outdated',
        'disabled',
        'in_segments',
    ]

    form_overrides = {
        'point_key': NullableStringField,
    }

    def scaffold_list_form(self, widget=None, validators=None):
        return super().scaffold_list_form(widget=SuggestWidget(), validators=validators)

    def get_list(self, *args, **kwargs):
        self._refresh_filters_cache()
        return super().get_list(*args, **kwargs)

    @expose('/rasp/<string(length>1):point_key>/edit', methods=('GET',))
    def rasp_edit(self, point_key):
        return jsonify(RaspAdminPointAdapter.edit(point_key))

    @expose('/rasp/<string(length>0):i>/<any(create_station, create_settlement):action>', methods=('POST',))
    def rasp_create(self, i, action):
        point_matching = self.get_one(i)
        if point_matching is None:
            raise exceptions.NotFound()
        if action == 'create_station':
            resp = RaspAdminPointAdapter.create_station(point_matching)
        elif action == 'create_settlement':
            resp = RaspAdminPointAdapter.create_settlement(point_matching)
        else:
            raise exceptions.NotFound()
        self.session.commit()
        return jsonify(resp)

    @expose('/ajax/lookup/', methods=('GET',))
    def ajax_lookup(self):
        try:
            query = request.args.get('query', '')

            point_key_pattern = r'^[{}{}]\d+$'.format(PointType.STATION.prefix, PointType.SETTLEMENT.prefix)
            point_key_lookup = re_search(point_key_pattern, query)
            result = []
            if point_key_lookup:
                resp = SuggestClient.get_by_point_key(query)
                if resp:
                    result.append(resp)
            else:
                offset = int(request.args.get('offset', '0'))
                limit = int(request.args.get('limit', '10'))
                resp = SuggestClient.search(query, limit=limit + offset)
                result = resp[offset: limit + offset]
            result = [(s['point_key'], '{point_key}: {full_title}'.format(**s)) for s in result]
        except (ValueError, KeyError) as e:
            logging.error('Suggest fails with: %r', e)
            raise exceptions.InternalServerError('Suggest format error')
        return jsonify(result)

    def render(self, template, **kwargs):
        if template == self.list_template:
            kwargs['matching_changes_count'] = self.session.query(func.count(MatchingChange.point_key)).scalar()
            kwargs['is_admin'] = ROLES.Admin.permission.allows(g.identity)
        elif template == self.details_template:
            model = kwargs['model']
            kwargs['logs_url'] = self.get_url('pointmatchinglog.index_view',
                                              flt1_supplier_point_id_equals=model.supplier_point_id)
        return super().render(template, **kwargs)

    def on_model_change(self, form, model, is_created):
        model.updated_by_login = g.identity.id
        super().on_model_change(form, model, is_created)

    def update_model(self, form, model):
        if 'point_key' in form.data:
            if form.data['point_key']:
                self.session.add(MatchingChange(supplier_id=model.supplier_id, point_key=form.data['point_key']))
            if model.point_key:
                self.session.add(MatchingChange(supplier_id=model.supplier_id, point_key=model.point_key))
        return super().update_model(form, model)

    @expose('/flush_matching_changes/', methods=('POST',))
    def flush_matching(self):
        if not ROLES.Admin.permission.allows(g.identity):
            flash("You do not have permission to perform this operation!", 'error')
        else:
            matchings = self.session.query(MatchingChange).join(MatchingChange.supplier).all()

            invalidate_data = defaultdict(set)
            if matchings:
                for dirty_matching in matchings:
                    invalidate_data[dirty_matching.supplier.code].add(dirty_matching.point_key)

            mapping_changes = json.dumps({'mapping_changes': {k: list(v) for k, v in invalidate_data.items()}})
            r = requests.post(Settings.Meta.URL + Settings.Meta.INVALIDATE_CACHE, data=mapping_changes)
            if r.status_code != requests.codes.ok:
                flash("Meta error: {}".format(r.text), 'error')
                return redirect(self.get_url('.index_view'))

            self.session.query(MatchingChange).delete()
            self.session.commit()
            flash("Done. Cache has been resetted.")

        return redirect(self.get_url('.index_view'))
