import re
import logging
from enum import Enum

import requests
from flask import request
from werkzeug import exceptions

from travel.rasp.bus.settings import Settings


class PointType(Enum):
    STATION = ('s', 'station')
    SETTLEMENT = ('c', 'settlement')

    def __init__(self, prefix, full):
        self.prefix = prefix
        self.full = full


class RaspAdminPointAdapter:
    """
    Helper for interacting with rasp admin
    """

    RASP_ADMIN_ADD_URL_PATTERN = '{url}/admin/www/{tp}/add/'
    RASP_ADMIN_CHANGE_URL_PATTERN = '{url}/admin/www/{tp}/{i}/change/'

    @staticmethod
    def _point_key_parse(point_key):
        if point_key.startswith(PointType.STATION.prefix):
            tp = PointType.STATION
        elif point_key.startswith(PointType.SETTLEMENT.prefix):
            tp = PointType.SETTLEMENT
        else:
            raise exceptions.BadRequest(f'Unknown point type for "{point_key}"')
        return tp, point_key[1:]

    @staticmethod
    def _response(url, point_key):
        return {'url': url, 'point_key': point_key}

    @classmethod
    def _create(cls, point_matching, point_type):
        url = cls.RASP_ADMIN_ADD_URL_PATTERN.format(
            url=Settings.RaspAdmin.URL,
            tp=point_type.full,
        )
        if 'Session_id' not in request.cookies:
            raise exceptions.SecurityError(
                'No session_id. In case of development environment authorize with any *.yandex-team.ru service')

        with requests.Session() as s:
            cookies = requests.cookies.RequestsCookieJar()
            cookies['Session_id'] = request.cookies['Session_id']
            cookies.update(request.cookies)

            csrf_response = s.get(url, cookies=cookies)
            csrf_response.raise_for_status()
            csrftoken = s.cookies.get('csrftoken')
            if not csrftoken:
                raise exceptions.SecurityError(
                    'Can not obtain csrftoken for {}'.format(Settings.RaspAdmin.URL))

            post_data = {
                'csrfmiddlewaretoken': csrftoken,
                '_continue': 'yes',
                'title': point_matching.title,
                'time_zone': 'Europe/Moscow',
            }
            if all((point_matching.latitude, point_matching.longitude)):
                post_data.update({
                    'latitude': point_matching.latitude,
                    'longitude': point_matching.longitude,
                })
            if point_type == PointType.STATION:
                post_data.update({
                    'majority': '4',
                    't_type': '3',
                    'station_type': '11',
                    'show_tablo_stat': 'on',
                    'stationterminal_set-TOTAL_FORMS': '3',
                    'stationterminal_set-INITIAL_FORMS': '0',
                    'stationterminal_set-MIN_NUM_FORMS': '0',
                    'stationterminal_set-MAX_NUM_FORMS': '1000',
                    'code_set-TOTAL_FORMS': '3',
                    'code_set-INITIAL_FORMS': '0',
                    'code_set-MIN_NUM_FORMS': '0',
                    'code_set-MAX_NUM_FORMS': '1000',
                    'stationphone_set-TOTAL_FORMS': '3',
                    'stationphone_set-INITIAL_FORMS': '0',
                    'stationphone_set-MIN_NUM_FORMS': '0',
                    'stationphone_set-MAX_NUM_FORMS': '1000',
                    'is_searchable_from': 'on',
                    'is_searchable_to': 'on',
                    'in_station_schedule': 'on',
                    'in_thread': 'on',
                    'use_in_departure_forecast': 'on',
                    'use_direction': 'subdir',
                    'show_settlement': 'on',
                    'show_mode': 'block',
                })
            elif point_type == PointType.SETTLEMENT:
                post_data.update({
                    'suggest_order': '100',
                    'use_in_suburban_app_suggests': 'no',
                    'related_settlement-TOTAL_FORMS': '1',
                    'related_settlement-INITIAL_FORMS': '0',
                    'related_settlement-MIN_NUM_FORMS': '0',
                    'related_settlement-MAX_NUM_FORMS': '1000',
                })
            else:
                raise exceptions.InternalServerError('Unknown PointType')

            push_response = s.post(
                url, headers=dict(Referer=url),
                data=post_data, cookies=cookies,
            )
            push_response.raise_for_status()

            found_id = re.search('\/(\d+)\/change', push_response.url)
            if not found_id:
                logging.error('Post model failed. Rasp Admin has changed form?')
                raise exceptions.BadRequest('Bad response url: {}'.format(push_response.url))

            logging.info('Created %r at %r', point_type.full, push_response.url)

            point_key = point_type.prefix + found_id[1]
            point_matching.point_key = point_key

            return cls._response(url=push_response.url, point_key=point_key)

    @classmethod
    def create_station(cls, point_matching):
        return cls._create(point_matching=point_matching, point_type=PointType.STATION)

    @classmethod
    def create_settlement(cls, point_matching):
        return cls._create(point_matching=point_matching, point_type=PointType.SETTLEMENT)

    @classmethod
    def edit(cls, point_key):
        point_type, i = cls._point_key_parse(point_key)
        url = cls.RASP_ADMIN_CHANGE_URL_PATTERN.format(
            url=Settings.RaspAdmin.URL,
            tp=point_type.full, i=i
        )
        return cls._response(url=url, point_key=point_key)
