# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

import requests
from requests.exceptions import RequestException
from six.moves.urllib_parse import urljoin

from travel.rasp.bus.api.connectors.entities.endpoint import Endpoint
from travel.rasp.bus.api.connectors.fields.validation import ValidationErrorBundle
from travel.rasp.bus.settings import Settings


class MetaClientError(Exception):
    pass


class TransportError(MetaClientError):
    pass


class ValidationError(MetaClientError):
    pass


class MetaClient(object):
    """Client for Meta and connectors (api3, api)"""

    @classmethod
    def get(cls, path, entity, as_list=True, raise_on_exception=True):
        data = cls.simple_get(path, raise_on_exception=raise_on_exception)
        if data is None:
            return
        try:
            if as_list:
                for d in data:
                    yield entity.init(d)
            else:
                yield entity.init(data)
        except (ValidationErrorBundle, ValueError) as e:
            logging.error('Validation error for %s: %s', path, e)
            if raise_on_exception:
                raise ValidationError(str(e))

    @classmethod
    def simple_get(cls, path, raise_on_exception=True):
        url = urljoin(Settings.Meta.URL, path)
        try:
            resp = requests.get(url)
            resp.raise_for_status()
        except RequestException as e:
            logging.error('Transport error for %s: %s', url, e)
            if raise_on_exception:
                raise TransportError(str(e))
            return None
        return resp.json()

    @classmethod
    def endpoints(cls, supplier_code, raise_on_exception=True):
        return cls.get(supplier_code + '/rides/endpoints', Endpoint,
                       as_list=True, raise_on_exception=raise_on_exception)

    @classmethod
    def raw_segments(cls, supplier_code, raise_on_exception=True):
        return cls.simple_get(supplier_code + '/rides/raw_segments', raise_on_exception=raise_on_exception)
