# -*- coding: utf-8 -*-
from flask_restful import fields


__all__ = [
    'Validatable',
    'ValidationError',
    'ValidationWarning',
    'ValidationErrorBundle',
    'validate'
]


class Validatable(object):
    def validate(self, value, path):
        raise NotImplementedError

    @staticmethod
    def format_msg(path, msg=""):
        return "{} is not valid: {}".format('.'.join(path), msg)

    @staticmethod
    def invalid_type(required, got):
        return "required {}, got {}".format(required, got)


class ValidationError(Exception):
    def __init__(self, path=[], msg=""):
        if path:
            msg = Validatable.format_msg(path, msg)
        super(ValidationError, self).__init__(msg)


class ValidationWarning(ValidationError):
    pass


class ValidationErrorBundle(Exception):
    def __init__(self, instance, exceptions):
        self.instance = instance
        self.exceptions = exceptions

    def __str__(self):
        return '\n'.join(map(str, self.exceptions))


def validate(field, data, path):
    if isinstance(field, Validatable):
        return field.validate(data, path)
    if data is None:
        return [ValidationWarning(path, "must not be None")]

    if isinstance(field, fields.Nested):
        field = field.nested
    elif isinstance(field, fields.List):
        if not isinstance(data, list):
            return [ValidationWarning(path, Validatable.invalid_type(list, type(data)))]
        field = field.container

    if isinstance(data, (list, tuple)):
        return [x for v in data for x in validate(field, v, path)]
    if isinstance(field, dict):
        if not isinstance(data, dict):
            return [ValidationWarning(path, Validatable.invalid_type(dict, type(data)))]
        return [x for k, v in field.items() for x in validate(v, data.get(k), path + [k])]
    return []
