import logging

from concurrent.futures import TimeoutError
from datetime import datetime, timedelta
from os import getpid
from typing import Optional

from kikimr.public.sdk.python.persqueue import grpc_pq_streaming_api as pqlib
from kikimr.public.sdk.python.persqueue.auth import OAuthTokenCredentialsProvider
from kikimr.public.sdk.python.persqueue.errors import SessionFailureResult

from travel.rasp.bus.settings import Settings


logger = logging.getLogger(__name__)


class LogBrokerClient:
    RECONNECT_ATTEMPT_SECS = 5 * 60
    TIMEOUT_SECS = 5

    def __init__(self, reconnect_attempt_timeout: Optional[int] = None, timeout: Optional[int] = None) -> None:
        self._enabled = Settings.LogBroker.ENABLED
        if not self._enabled:
            return
        self._reconnect_attempt_timeout = reconnect_attempt_timeout or self.RECONNECT_ATTEMPT_SECS
        self._timeout = timeout or self.TIMEOUT_SECS
        self._credentials_provider = OAuthTokenCredentialsProvider(Settings.LogBroker.TOKEN.encode())
        self._api = None
        self._producer = None
        self._api_started = False
        self._connection_attempt_at = None
        self._seq_id = 0

    def _get_api(self):
        if not self._api_started:
            if self._api is None:
                self._api = pqlib.PQStreamingAPI(Settings.LogBroker.ENDPOINT, Settings.LogBroker.PORT)
            now = datetime.now()
            if self._connection_attempt_at is not None \
                    and self._connection_attempt_at + timedelta(seconds=self._reconnect_attempt_timeout) > now:
                return None
            self._connection_attempt_at = now
            try:
                self._api.start().result(timeout=self._timeout)
            except TimeoutError:
                logger.error('Can not connect to LogBroker. Timeout')
                return None
            self._api_started = True
        return self._api

    def _get_producer(self):
        if self._producer is None or self._producer.stop_future.done():
            lb_api = self._get_api()
            if lb_api is None:
                return None
            source_id = '{}:{}'.format(Settings.LogBroker.SOURCE_ID_PREFIX, getpid())
            self._producer = self._api.create_retrying_producer(
                pqlib.ProducerConfigurator(Settings.LogBroker.ADMIN_ACCESS_TOPIC, source_id.encode()),
                credentials_provider=self._credentials_provider
            )
            try:
                start_result = self._producer.start().result(timeout=self._timeout)
            except TimeoutError:
                logger.error('Can not start producer. Timeout')
                return
            if isinstance(start_result, SessionFailureResult):
                logger.error('Producer can not establish session with LogBroker: %s', start_result)
                return None
        return self._producer

    @property
    def enabled(self) -> bool:
        return self._enabled

    def publish(self, msg: bytes) -> bool:
        if not self.enabled:
            logger.info('Can not publish to LogBroker. Disabled in settings module')
            return False
        producer = self._get_producer()
        if producer is None:
            return False
        self._seq_id += 1
        try:
            producer.write(self._seq_id, msg).result(timeout=self._timeout)
        except TimeoutError:
            logger.error('Can not publish to LogBroker. Write timeout')
            return False
        return True
