# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from flask_restful import abort, fields, marshal, reqparse, Resource
from sqlalchemy import and_, or_
from sqlalchemy.sql.expression import false

from travel.rasp.bus.db import session_scope
from travel.rasp.bus.db.models.carrier import Carrier
from travel.rasp.bus.db.models.carrier_matching import CarrierMatching
from travel.rasp.bus.db.models.supplier import Supplier
from travel.rasp.bus.morda_backend.views.organization import organization_fields

matching_key = {
    'carrierCode': fields.String(),
    'supplierCode': fields.String(),
}
matching = {
    'key': fields.Nested(matching_key),
    'carrier': fields.Nested(organization_fields),
    'supplier': fields.Nested(organization_fields),
}
raw_matching = {
    'code': fields.String(),
    'supplier_id': fields.Integer(),
    'carrier_id': fields.Integer(),
}

matching_get_parser = reqparse.RequestParser()
matching_get_parser.add_argument(
    'query', action='append',
    type=unicode,
    default=None,
    help='Format: [supplier_code/carrier_code]',
)
matching_post_parser = reqparse.RequestParser()
matching_post_parser.add_argument(
    'query',
    type=dict,
    default=None,
    location='json',
    help='Format: {"query":{"supplier_code1": ["carrier_code1", "carrier_code2"], "supplier_code2": [...], ...}',
)


class CarrierMatchingSearchRoute(Resource):
    def get(self):
        '''Search matching'''
        args = matching_get_parser.parse_args()
        queries = args['query']

        if not queries:
            abort(404, message='Matchings with queries {} do not exsist'.format(queries))

        supplier_carrier_code_pairs = [q.split('/') for q in queries]

        return self._get_carriers(supplier_carrier_code_pairs)

    def post(self):
        '''Search matching'''
        args = matching_post_parser.parse_args()
        json_queries = args['query']

        if not json_queries:
            abort(404, message='Matchings with queries {} do not exsist'.format(json_queries))

        supplier_carrier_code_pairs = [(supplier_code, carrier_code) for supplier_code, carrier_codes in json_queries.items()
                                       for carrier_code in carrier_codes]

        return self._get_carriers(supplier_carrier_code_pairs)

    def _get_carriers(self, supplier_carrier_code_pairs):
        filters = [
            and_(Supplier.code == supplier_code, CarrierMatching.code == carrier_code)
            for (supplier_code, carrier_code) in supplier_carrier_code_pairs
        ]

        with session_scope() as session:
            query = (
                session.query(Carrier, CarrierMatching.code, Supplier)
                .join(CarrierMatching)
                .join(Supplier)
                .filter(or_(*filters))
                .filter(Supplier.hidden == false(), Carrier.hidden == false())
            )
            result = query.all()

            return marshal([
                {
                    'carrier': c,
                    'supplier': s,
                    'key': {
                        'carrierCode': carrier_code,
                        'supplierCode': s.code,
                    }
                } for (c, carrier_code, s) in result
            ], matching)


class CarrierMatchingListRoute(Resource):
    def get(self):
        '''List all carrier matching'''
        with session_scope() as session:
            items = marshal(session.query(CarrierMatching).all(), raw_matching)
        return items
