# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging

from collections import defaultdict

from travel.rasp.bus.db import session_scope
from travel.rasp.bus.db.models.admin_user import AdminUser
from travel.rasp.bus.db.models.supplier import Supplier

from travel.rasp.bus.scripts.automatcher import policy
from travel.rasp.bus.scripts.automatcher.scenario import get_scenarios_by_names


log = logging.getLogger('automatcher')


class Stats:
    def __init__(self):
        self.results = defaultdict(lambda: defaultdict(int))
        self.actions = [('supplier', 'supplier_point_id', 'scenario', 'point_key', 'auto_result')]

    def print_results(self):
        for supplier_code, scenarios in self.results.items():
            for scenario, value in scenarios.items():
                log.info('supplier: %s, scenario: %s, result: %s', supplier_code, scenario, value)

    def get_actions(self):
        return self.actions

    def add_action(self, supplier, point, scenario, value):
        self.results[supplier.code][scenario.name] += 1
        self.actions.append((supplier.code, point.supplier_point_id, scenario.name, point.point_key, value))


class AutoMatcher(object):
    """
    Params:
    scenarios_names - list of scenarios to run, must be not empty
    suppliers_names - list of suppliers, if empty - run for all suppliers
    dry_run - writes result of first successful scenario for point in DB if False,
        dont write results to DB if True
    run_all - can change MatchedPolicy scope if True, in this case scenarios with
        matched_policy = MatchedPolicy.NOT_MATCHED will be applied to MATCHED points,
        dry_run will be forced, and results will be logged for all scenarios for each point, not only first
        successful as usual.
    """
    def __init__(self, scenarios_names, suppliers_names, params, login, dry_run=True, run_all=False):
        self.supplier_names = suppliers_names
        self.user = self._get_user(login)
        self.dry_run = dry_run
        self.run_all = run_all
        if self.run_all:
            self.dry_run = True
            log.info('Dry run is forced by RUN_ALL!')
        self.stats = Stats()

        common_params = {
            'dry': self.dry_run,
        }
        params['common'] = common_params
        log.info('Start loading scenarios')
        self.scenarios = self.load_scenarios(scenarios_names, params)
        log.info('Load scenarios: done')
        if not self.scenarios:
            raise ValueError('empty list of scenarios')

    def run(self):
        log.info('Start automatching')
        if self.dry_run:
            log.info('Dry run!')

        with session_scope() as session:
            if not self.supplier_names:
                suppliers = session.query(Supplier).all()
            else:
                suppliers = session.query(Supplier).filter(Supplier.code.in_(self.supplier_names)).all()
            log.info('suppliers: %s', ', '.join(sp.code for sp in suppliers))
            log.info('scenarios: %s', ', '.join(sc.name for sc in self.scenarios))
            for supplier in suppliers:
                log.info('Supplier %s', supplier.code)
                for point in supplier.point_matchings:
                    for scenario in self.scenarios:
                        if self._check_policy(point, scenario):
                            has_result, value = scenario.run(point)
                            if has_result:
                                log.info('    point %d, %s: got result %s by matcher %s',
                                         point.supplier_id, point.supplier_point_id, value, scenario.name)
                                self.stats.add_action(supplier, point, scenario, value)
                                if self.run_all is False:
                                    point.point_key = value
                                    point.updated_by = self.user
                                    break
            if self.dry_run:
                log.info('Dry run, no commit. Found matchings:')
                self.stats.print_results()
                session.rollback()
            else:
                log.info('Found and updated matchings:')
                self.stats.print_results()
                session.commit()
        for scenario in self.scenarios:
            scenario.postprocess()

    def _check_policy(self, point, scenario):
        return policy.check_policies(point, scenario, self.run_all)

    def get_stats(self):
        stats = {'automatcher': self.stats.get_actions()}
        for scenario in self.scenarios:
            scenario_report = scenario.get_report()
            if scenario_report:
                stats[scenario.name] = scenario_report
        return stats

    def load_scenarios(self, scenarios_names, params):
        return get_scenarios_by_names(scenarios_names, params)

    def _get_user(self, login):
        with session_scope() as session:
            user = session.query(AdminUser).get(login)
            if user is None:
                raise ValueError('Invalid user name: {}'.format(login))
            return user
