# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from enum import Enum

from travel.rasp.bus.db.models.matching import PointType


class MatchedPolicy(Enum):
    NOT_MATCHED = 1
    MATCHED = 2


class TypePolicy(Enum):
    TYPE_CITY = 1
    TYPE_STATION = 2
    TYPE_ALL = 3

    def is_matched(self, point):
        if self is TypePolicy.TYPE_STATION:
            return point.type is PointType.STATION
        if self is TypePolicy.TYPE_CITY:
            return point.type is PointType.CITY
        return True


def _get_point_policy(point):
    if point.point_key is None:
        return MatchedPolicy.NOT_MATCHED
    else:
        return MatchedPolicy.MATCHED


def _check_matching_status_policy(point, scenario, run_all):
    point_policy = _get_point_policy(point)
    if scenario.matched_policy == MatchedPolicy.MATCHED:
        return point_policy == scenario.matched_policy
    elif not run_all:
        return point_policy == scenario.matched_policy
    return True


def _check_supplier_policy(point, scenario, run_all):
    if scenario.supplier is not None:
        return scenario.get_scenario_supplier_id() == point.supplier_id
    return True


def _check_point_type_policy(point, scenario, run_all):
    if scenario.point_type_policy is None:
        raise ValueError("unexpected unset point type policy at class {}".format(scenario.__class__.__name__))

    return scenario.point_type_policy.is_matched(point)


POLICY_CHECKS = [
    _check_matching_status_policy,
    _check_point_type_policy,
    _check_supplier_policy,
]


def check_policies(point, scenario, run_all):
    for policy_check in POLICY_CHECKS:
        if not policy_check(point, scenario, run_all):
            return False
    return True
