from collections import defaultdict
from dataclasses import dataclass
from datetime import datetime, timedelta
from typing import List, DefaultDict


@dataclass
class State:
    datetime: datetime
    requests_number: int
    cache_hit_number: int
    cache_miss_number: int

    @staticmethod
    def new_zeroed_state(time):
        return State(requests_number=0,
                     cache_hit_number=0,
                     cache_miss_number=0,
                     datetime=time)


@dataclass
class Statistic:
    datetimes: List[datetime]
    requests_numbers: List[int]
    cache_hit_numbers: List[int]
    cache_miss_numbers: List[int]
    cache_entries_age_distribution: DefaultDict[timedelta, int]

    @staticmethod
    def new_empty_statistic():
        return Statistic(datetimes=[],
                         requests_numbers=[],
                         cache_hit_numbers=[],
                         cache_miss_numbers=[],
                         cache_entries_age_distribution=defaultdict(int))


def round_time_floor(time, period):
    return time - timedelta(minutes=time.minute % period, seconds=time.second, microseconds=time.microsecond)


def round_delta_time_floor(delta, period):
    period_in_seconds = period * 60
    return timedelta(seconds=(delta.total_seconds() -
                              delta.total_seconds() % period_in_seconds))


class Cache:
    ALL_CACHE_KEY = "all"
    DATE = {}


class CacheEmulator:
    def __init__(self, first_entry_time, segments, ttl=60, norm=15):
        self.norm = timedelta(minutes=norm)
        self.cache_age_bucket_size = norm
        self.ttl = timedelta(minutes=ttl)
        self.segments = segments
        self.current_suppliers = set()
        if first_entry_time is not None:
            self.current_time = first_entry_time
            start_time = round_time_floor(self.current_time, self.norm.total_seconds() / 60)
            self.state = State.new_zeroed_state(start_time)
            self.last_response_time = start_time
            self.statistic = Statistic.new_empty_statistic()
            self.suppliers_statistics = {}
            self.suppliers_states = {}
            for supplier in self.segments.keys():
                self.suppliers_statistics[supplier] = Statistic.new_empty_statistic()
                self.suppliers_states[supplier] = State.new_zeroed_state(start_time)

    def is_cache_not_rotten(self, entry, supplier):
        return entry.datetime - Cache.DATE[(entry.request, supplier)] < self.ttl

    def is_ready_to_save_state(self):
        return round_time_floor(self.current_time, self.norm.total_seconds() / 60) > self.state.datetime

    def find_suppliers_for_request(self, request):
        return set(filter(lambda s: (request.direction.from_id in self.segments[s] and
                                     request.direction.to_id in self.segments[s][request.direction.from_id]),
                          self.segments.keys()))

    def add_state_to_statistic(self, state, statistic):
        statistic.cache_hit_numbers.append(state.cache_hit_number)
        statistic.cache_miss_numbers.append(state.cache_miss_number)
        statistic.datetimes.append(state.datetime)
        statistic.requests_numbers.append(state.requests_number)

    def add_suppliers_states_to_statistic(self):
        for supplier in self.segments.keys():
            self.add_state_to_statistic(self.suppliers_states[supplier], self.suppliers_statistics[supplier])

    def update_request_number(self, state):
        state.requests_number += 1

    def update_cache_hit_number(self, state):
        state.cache_hit_number += 1

    def update_cache_miss_number(self, state):
        state.cache_miss_number += 1

    def update_cache_age_distribution(self, entry, supplier, statistic):
        age = round_delta_time_floor(entry.datetime - Cache.DATE[(entry.request, supplier)], self.cache_age_bucket_size)
        statistic.cache_entries_age_distribution[age] += 1

    def save_states(self):
        self.last_response_time = self.state.datetime
        time = round_time_floor(self.current_time, self.norm.total_seconds() / 60)
        self.add_state_to_statistic(self.state, self.statistic)
        self.state = State.new_zeroed_state(time)
        self.add_suppliers_states_to_statistic()
        for supplier in self.segments.keys():
            self.suppliers_states[supplier] = State.new_zeroed_state(time)

    def emulate_rides_search(self, entry, supplier, state, statistic):
        self.update_request_number(state)
        if (entry.request, supplier) in Cache.DATE and self.is_cache_not_rotten(entry, supplier):
            self.update_cache_hit_number(state)
            self.update_cache_age_distribution(entry, supplier, statistic)
        else:
            Cache.DATE[(entry.request, supplier)] = entry.datetime
            self.update_cache_miss_number(state)

    def run(self, entry, suppliers=None):
        self.current_time = entry.datetime
        if not suppliers:
            self.current_suppliers = self.find_suppliers_for_request(entry.request)
        else:
            self.current_suppliers = suppliers
        if self.is_ready_to_save_state():
            self.save_states()
        self.emulate_rides_search(entry, Cache.ALL_CACHE_KEY, self.state, self.statistic)
        for supplier in self.current_suppliers:
            self.emulate_rides_search(entry, supplier,
                                      self.suppliers_states[supplier],
                                      self.suppliers_statistics[supplier])
