from dataclasses import dataclass
from datetime import datetime
import re
from urllib import parse
from typing import Optional


@dataclass(eq=True, frozen=True)
class Direction:
    from_id: str
    to_id: str

    @staticmethod
    def new_direction(from_id, to_id):
        return Direction(
            from_id=from_id,
            to_id=to_id,
        )


@dataclass(eq=True, frozen=True)
class Request:
    direction: Direction
    date: Optional[datetime]

    @staticmethod
    def new_request(request_string):
        params = parse.parse_qs(parse.urlsplit(request_string).query)
        date = None
        if re.fullmatch(r'\d{4}-\d{2}-\d{2}', params['date'][0]):
            date = datetime.strptime(params['date'][0], '%Y-%m-%d').date()
        elif re.fullmatch(r'\d{2}.\d{2}.\d{2}', params['date'][0]):
            date = datetime.strptime(params['date'][0], '%d.%m.%y').date()
        return Request(
            direction=Direction.new_direction(
                from_id=params['from-id'][0],
                to_id=params['to-id'][0]),
            date=date
        )


@dataclass
class LogEntry:
    datetime: datetime
    request: Request

    @staticmethod
    def new_log_entry(data):
        return LogEntry(
            datetime=datetime.strptime(data['datetime'], '%Y-%m-%dT%H:%M:%S+00:00'),
            request=Request.new_request(data['request'])
        )


def build_log_entries(raw_data_iterator):
    return list(map(LogEntry.new_log_entry, raw_data_iterator))
