# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import argparse
import logging.config


from travel.rasp.bus.scripts.cache_reseter.cache_reseter import cache_reseter
from travel.rasp.bus.scripts.library.utils import LOGGING_CONFIG


def parse_args():
    parser = argparse.ArgumentParser(description='The tool that helps to reset Yandex.Bus cache data')
    subparsers = parser.add_subparsers(dest='cmd', required=True)
    reset_segments_for_cmd = subparsers.add_parser(
        'segments', help='Reset the segment cache'
    )
    reset_segments_for_cmd.add_argument(
        '--suppliers', '-s',
        action='append',
        default=[],
        help='a suppliers code',
        type=str,
    )
    reset_segments_for_cmd.add_argument(
        '--dry',
        action='store_const',
        const=True,
        default=False,
        help='Do not change any data',
    )

    search_data_by_point_cmd = subparsers.add_parser(
        'search-data-by-point',
        help='Reset the search data cache by point',
    )
    search_data_by_point_cmd.add_argument(
        '--supplier', '-s',
        help='a supplier code',
        type=str,
        required=True,
    )
    search_data_by_point_cmd.add_argument(
        '--point', '-p',
        help='a rasp point code',
        type=str,
        required=True,
    )
    search_data_by_point_cmd.add_argument(
        '--dry',
        action='store_const',
        const=True,
        default=False,
        help='Do not change any data',
    )

    search_data_by_direction_cmd = subparsers.add_parser(
        'search-data-by-direction',
        help='Reset the search data cache by direction',
    )
    search_data_by_direction_cmd.add_argument(
        '--supplier', '-s',
        help='a supplier code',
        type=str,
        required=True,
    )
    search_data_by_direction_cmd.add_argument(
        '--from-point', '-f',
        help='a departure rasp point code',
        type=str,
        required=True,
    )
    search_data_by_direction_cmd.add_argument(
        '--to-point', '-t',
        help='an arrival rasp point code',
        type=str,
        required=True,
    )
    search_data_by_direction_cmd.add_argument(
        '--two-way', '--tw',
        action='store_const',
        const=True,
        default=False,
        help='Do the same operation for the backward route',
    )
    search_data_by_direction_cmd.add_argument(
        '--dry',
        action='store_const',
        const=True,
        default=False,
        help='Do not change any data',
    )

    search_data_by_partucular_direction_cmd = subparsers.add_parser(
        'search-data-by-particular-direction',
        help='Reset the search data cache by full direction',
    )
    search_data_by_partucular_direction_cmd.add_argument(
        '--supplier', '-s',
        type=str,
        help='a supplier code',
        required=True,
    )
    search_data_by_partucular_direction_cmd.add_argument(
        '--from-point', '-f',
        help='a departure rasp point code',
        type=str,
        required=True,
    )
    search_data_by_partucular_direction_cmd.add_argument(
        '--to-point', '-t',
        help='an arrival rasp point code',
        type=str,
        required=True,
    )
    search_data_by_partucular_direction_cmd.add_argument(
        '--when', '-w',
        help='an departure date',
        type=str,
        required=True,
    )
    search_data_by_partucular_direction_cmd.add_argument(
        '--dry',
        action='store_const',
        const=True,
        default=False,
        help='Do not change any data',
    )

    return parser.parse_args()


if __name__ == '__main__':
    logging.config.dictConfig(LOGGING_CONFIG)
    args = parse_args()

    cmd = args.cmd
    if cmd == 'segments':
        cache_reseter.reset_segments_for(
            suppliers=args.suppliers,
            dry=args.dry,
        )
    elif cmd == 'search-data-by-point':
        cache_reseter.reset_search_data_by_point(
            supplier=args.supplier,
            point_code=args.point,
            dry=args.dry,
        )
    elif cmd == 'search-data-by-direction':
        cache_reseter.reset_search_data_by_direction(
            supplier=args.supplier,
            from_point=args.from_point,
            to_point=args.to_point,
            two_way=args.two_way,
            dry=args.dry,
        )
    elif cmd == 'search-data-by-particular-direction':
        cache_reseter.reset_search_data(
            supplier=args.supplier,
            from_point=args.from_point,
            to_point=args.to_point,
            when=args.when,
            dry=args.dry,
        )
    else:
        raise NotImplementedError('This sub command is not implemented')
