# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

from abc import ABCMeta, abstractmethod

import redis
from redis.sentinel import Sentinel

from travel.rasp.bus.settings import Settings


READ_CLIENT_NAME = 'slave'
WRITE_CLIENT_NAME = 'master'


class IRedisProvider:
    __metaclass__ = ABCMeta

    @abstractmethod
    def get_read_client(self):
        raise NotImplementedError

    @abstractmethod
    def get_write_client(self):
        raise NotImplementedError


class RedisProvider(IRedisProvider):
    def _get_client(self, role):
        cache_settings = Settings.Cache

        host = cache_settings.REDIS_HOST
        if host:
            return redis.Redis(
                host=host,
                port=cache_settings.REDIS_PORT,
                socket_timeout=cache_settings.REDIS_SOCKET_TIMEOUT.total_seconds(),
                db=cache_settings.REDIS_DB
            )

        sentinel_name = cache_settings.REDIS_SENTINEL_NAME
        if sentinel_name:
            s = Sentinel(
                sentinels=cache_settings.REDIS_SENTINEL_NODES,
                socket_timeout=cache_settings.REDIS_SOCKET_TIMEOUT.total_seconds(),
                db=cache_settings.REDIS_DB,
                password=cache_settings.REDIS_SENTINEL_PASSWORD
            )
            if role == WRITE_CLIENT_NAME:
                return s.master_for(sentinel_name)
            if role == READ_CLIENT_NAME:
                return s.slave_for(sentinel_name)

        raise Exception('Can not create a redis client')

    def get_read_client(self):
        return self._get_client(READ_CLIENT_NAME)

    def get_write_client(self):
        return self._get_client(WRITE_CLIENT_NAME)


redis_provider = RedisProvider()
