import dataclasses
import datetime
import logging
import typing

from travel.rasp.bus.db import session_scope
from travel.rasp.bus.db.models.order import Order
from travel.rasp.bus.db.models.order_log_entry import OrderLogEntry
from travel.rasp.bus.library import trust_client
from travel.rasp.bus.settings import Settings

log = logging.getLogger(__name__)


@dataclasses.dataclass(frozen=True)
class OrderPurchase:
    order_id: str
    service_id: int
    purchase_token: str


class TrustClientFactory:
    _trust_service_tokens = None

    @classmethod
    def get_client(cls, service_id: int) -> trust_client.Client:
        tokens = cls._trust_service_tokens
        if tokens is None:
            tokens = cls._trust_service_tokens = Settings.Trust.get_service_tokens()
        return trust_client.Client(Settings.Trust.URL, tokens[str(service_id)])


def list_uncleared_purchases(
    min_creation_ts: datetime.datetime, max_creation_ts: typing.Optional[datetime.datetime] = None
) -> typing.Tuple[OrderPurchase, ...]:
    if max_creation_ts is not None and max_creation_ts <= min_creation_ts:
        raise ValueError("max_creation_ts should be greater than min_creation_ts")

    log.info("Collecting orders from %r to %r", min_creation_ts, max_creation_ts)

    creation_ts_condition = Order.creation_ts.isnot(None) & (Order.creation_ts >= min_creation_ts)
    if max_creation_ts is not None:
        creation_ts_condition &= Order.creation_ts < max_creation_ts

    with session_scope() as session:
        query = (
            session.query(Order.id, Order.booking["serviceId"], Order.purchase["purchase_token"])
            .filter(
                creation_ts_condition,
                Order.status == "confirmed",
                ~Order.log_entries.any(OrderLogEntry.status == "cleared"),  # type: ignore
            )
            .order_by(Order.creation_ts)
        )
        return tuple(OrderPurchase(*row) for row in query)


def check_and_clear_payments(
    min_creation_ts: datetime.datetime,
    max_creation_ts: typing.Optional[datetime.datetime] = None,
    dry_run: bool = False,
) -> None:
    purchases = list_uncleared_purchases(min_creation_ts, max_creation_ts)
    log.info("Found %s uncleared orders", len(purchases))

    for purchase in purchases:
        log.info("Processing %r...", purchase)
        try:
            trust = TrustClientFactory.get_client(purchase.service_id)
            payment_status = trust.get_payment_data(purchase.purchase_token)["payment_status"]
            if payment_status != "cleared":
                log.info("Payment status is %s", payment_status)
                if not dry_run:
                    log.info("Clearing payment...")
                    trust.clear_payment(purchase.purchase_token)
        except Exception:
            log.exception("Exception during processing of %r", purchase)

    log.info("Done")
