import logging
import typing
from datetime import datetime, timedelta

import click
import isodate

from travel.rasp.bus.scripts.clearing_checker import check_and_clear_payments

log = logging.getLogger(__name__)


@click.command()
@click.option("--min-creation-ts", help="Minimum value of the order creation timestamp", type=isodate.parse_datetime)
@click.option("--max-creation-ts", help="Maximum value of the order creation timestamp", type=isodate.parse_datetime)
@click.option("--creation-ts-age", help="Age of processed orders (default is 1 hour)", type=isodate.parse_duration)
@click.option("-v", "--verbose", "logging_level", flag_value=logging.NOTSET)
@click.option("-q", "--quiet", "logging_level", flag_value=logging.WARNING)
@click.option("--dry-run", help="Dry run", is_flag=True)
def main(
    min_creation_ts: typing.Optional[datetime],
    max_creation_ts: typing.Optional[datetime],
    creation_ts_age: typing.Optional[timedelta],
    logging_level: typing.Optional[int],
    dry_run: bool,
) -> None:
    logging.basicConfig(level=(logging_level if logging_level is not None else logging.INFO))

    if min_creation_ts is None:
        min_creation_ts = datetime.utcnow() - (creation_ts_age if creation_ts_age is not None else timedelta(hours=1))
    elif creation_ts_age:
        raise click.UsageError("Only one of options allowed --min-creation-ts or --creation-ts-age")

    check_and_clear_payments(min_creation_ts=min_creation_ts, max_creation_ts=max_creation_ts, dry_run=dry_run)


if __name__ == "__main__":
    main()
