# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import datetime
import logging
import requests
import os
import tempfile

import unicodecsv as csv
import boto3
from boto3.exceptions import Boto3Error
from yql.api.v1.client import YqlClient
from yql.client.parameter_value_builder import YqlParameterValueBuilder
from six.moves.urllib_parse import urljoin
from travel.library.python.rasp_vault.api import get_secret

from travel.rasp.bus.settings import Settings


class FeedsUpdater(object):

    _RUSSIA_RASP_ID = 225
    _RUB_ID = 1
    _LOCAL_DIR = tempfile.gettempdir()
    _S3_DIR = 'feeds'
    _FEEDS_FN = 'feeds.csv'
    _DEFAULT_SETTLEMENT_IMAGE = 'https://s3.mds.yandex.net/travel.buses-public.production/feeds/settlement_cap.png'

    def __init__(self, days_count, orders_threshold, limit, default_settlement_image=None, dry_run=False):
        self._days_count, self._orders_threshold, self._limit = days_count, orders_threshold, limit
        self._yql_token = get_secret(Settings.YavKeys.YQL_TOKEN)
        self._s3_key, self._s3_secret = get_secret(Settings.YavKeys.S3_KEY), get_secret(Settings.YavKeys.S3_SECRET)
        self._s3_endpoint, self._s3_bucket = Settings.S3.ENDPOINT, Settings.S3.PUBLIC_BUCKET
        self._default_settlement_image = default_settlement_image or self._DEFAULT_SETTLEMENT_IMAGE
        self._dry_run = dry_run
        self._logger = logging.getLogger()

    def _get_directions_revenue(self):
        proxy = 'hahn'
        query = '''
            DECLARE $from_date AS Date;
            DECLARE $orders_threshold AS UInt64;
            DECLARE $default_settlement_image AS String;
            DECLARE $limit AS UInt64;

            SELECT d.from_settlement_id, d.to_settlement_id, d.orders_count, d.yandex_revenue,
                IF(i.Url IS NULL, $default_settlement_image, i.Url)
            FROM
            (
                SELECT SUM(total_agency_fee_amount + total_fee_amount) AS yandex_revenue,
                    COUNT(*) AS orders_count,
                    CAST(SUBSTRING(search_from_id, 1) AS UInt64) as from_settlement_id,
                    CAST(SUBSTRING(search_to_id, 1) AS UInt64) as to_settlement_id
                FROM `home/travel/prod/cpa/buses/orders`
                WHERE status='confirmed'
                    AND CAST(DateTime::FromSeconds(CAST(created_at AS Uint32)) AS Date) >= $from_date
                    AND SUBSTRING(search_from_id, 0, 1) = 'c'
                    AND SUBSTRING(search_to_id, 0, 1) = 'c'
                GROUP BY search_from_id, search_to_id
                HAVING COUNT(*) >= $orders_threshold
            ) AS d
            INNER JOIN
            (
                SELECT Id, GeoId
                FROM `home/travel/prod/rasp_dicts/latest/settlement`
            ) AS s
            ON (d.to_settlement_id = s.Id)
            LEFT JOIN
            (
                SELECT GeoId, Url
                FROM `home/travel/prod/config/region_images`
            ) AS i
            ON (s.GeoId = i.GeoId)
            LIMIT $limit
        '''

        params = {
            '$from_date': YqlParameterValueBuilder.make_date(datetime.date.today() - datetime.timedelta(days=self._days_count)),
            '$orders_threshold': YqlParameterValueBuilder.make_int64(self._orders_threshold),
            '$default_settlement_image': YqlParameterValueBuilder.make_string(self._default_settlement_image),
            '$limit': YqlParameterValueBuilder.make_int64(self._limit * 2)
        }

        with YqlClient(db=proxy, token=self._yql_token) as yql_client:
            request = yql_client.query(query, syntax_version=1)
            request.run(parameters=YqlParameterValueBuilder.build_json_map(params))

            for table in request:
                for row in table.get_iterator():
                    yield row

    def _get_api_popular_directions(self):
        url = urljoin(Settings.Backend.API_URL, 'popular')
        resp = requests.get(url, params=dict(limit=self._limit * 2))
        resp.raise_for_status()
        pop_dirs = resp.json().get('Directions')
        self._logger.info('Directions from backend/api/popular: {}'.format(len(pop_dirs)))
        return pop_dirs

    def _upload_to_s3(self, file_path):
        self._logger.info('Uploading to s3: endpoint={}, bucket={}, name={}'.format(
            self._s3_endpoint, self._s3_bucket, self._FEEDS_FN),
        )
        s3_client = boto3.client(
            's3',
            aws_access_key_id=self._s3_key,
            aws_secret_access_key=self._s3_secret,
            endpoint_url=self._s3_endpoint,
        )
        try:
            s3_client.upload_file(file_path, self._s3_bucket, os.path.join(self._S3_DIR, self._FEEDS_FN))
        except Boto3Error as e:
            self._logger.error('Can not upload to s3: {}', e)
            raise e
        self._logger.info('Done. Public link: {}/{}/{}/{}'.format(
            self._s3_endpoint, self._s3_bucket, self._S3_DIR, self._FEEDS_FN
        ))

    def run(self):
        revenues = {}
        for from_settlement_id, to_settlement_id, orders_count, yandex_revenue, settlement_image in self._get_directions_revenue():
            revenues[(from_settlement_id, to_settlement_id)] = (orders_count, yandex_revenue, settlement_image)
        self._logger.info('Directions from CPA: {}'.format(len(revenues)))

        pop_dirs = self._get_api_popular_directions()

        field_names = ['DirectionId*', 'FromSettlementTitleDefault', 'ToSettlementTitleDefault', 'url', 'LegacyUrl',
                       'OrdersCount', 'TotalRevenue', 'AvgRevenue', 'DirectionType', 'FromSettlementTitleGenitive',
                       'ToSettlementAccusativeCase', 'MinPrice', 'SearchPopularity', 'ToSettlementImageUrl']
        feeds = []
        for pop_dir in pop_dirs:
            try:
                frm, to = pop_dir['From'], pop_dir['To']

                key = frm['Id'], to['Id']
                if key not in revenues:
                    continue
                orders_count, yandex_revenue, settlement_image = revenues[key]
                if orders_count == 0:
                    continue

                minPrice = None
                for price in pop_dir['Prices']:
                    if price['Currency'] == self._RUB_ID:
                        minPrice = price['Amount'] / 10**price.get('Precision', 2)
                        break
                if minPrice is None:
                    continue

                feeds.append({
                    'DirectionId*': '{}--{}'.format(frm['Slug'], to['Slug']),
                    'FromSettlementTitleDefault': frm['TitleDefault'],
                    'ToSettlementTitleDefault': to['TitleDefault'],
                    'url': 'https://travel.yandex.ru/buses/{}--{}/?date=tomorrow'.format(frm['Slug'], to['Slug']),
                    'LegacyUrl': 'https://yandex.ru/bus/{}/{}'.format(frm['TitleDefault'], to['TitleDefault']),
                    'OrdersCount': orders_count,
                    'TotalRevenue': yandex_revenue,
                    'AvgRevenue': yandex_revenue / orders_count,
                    'DirectionType': 'RUSSIA' if frm.get('CountryId', self._RUSSIA_RASP_ID) == to.get('CountryId', self._RUSSIA_RASP_ID) == self._RUSSIA_RASP_ID else 'INTERNATIONAL',
                    'FromSettlementTitleGenitive': 'из {}'.format(frm['Title']['Ru']['Genitive']),
                    'ToSettlementAccusativeCase': '{} {}'.format(to['Title']['Ru']['LocativePreposition'], to['Title']['Ru']['Accusative']),
                    'MinPrice': minPrice,
                    'SearchPopularity': int(pop_dir['Weight']),
                    'ToSettlementImageUrl': settlement_image,
                })
            except KeyError as e:
                self._logger.warning('backend/api/popular key error: {}'.format(e))
                continue

            if len(feeds) == self._limit:
                break

        self._logger.info('Generated feeds: {}'.format(len(feeds)))

        if len(feeds) == 0:
            self._logger.info('No feeds generated. Will do nothing')
            raise ValueError('No feeds generated')

        report_path = os.path.join(self._LOCAL_DIR, self._FEEDS_FN)
        with open(report_path, 'wb') as f:
            writer = csv.DictWriter(f, fieldnames=field_names)
            writer.writeheader()
            writer.writerows(feeds)
        self._logger.info('Generated feeds saved as {}'.format(report_path))

        if not self._dry_run:
            self._upload_to_s3(report_path)
            os.remove(report_path)
            self._logger.info('Generated feeds removed')
