# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import calendar
import itertools
import locale
from collections import defaultdict
from datetime import datetime
from logging import getLogger

from travel.rasp.bus.db import session_scope
from travel.rasp.bus.db.models.order import Order
from travel.rasp.bus.registry.mail_sender import mail_sender
from travel.rasp.bus.scripts.order_report.utils import dump_csv, get_human_month

logger = getLogger('order_report')


def range_report(start_range, end_range):
    logger.info('Collecting data from %r to %r', start_range, end_range)
    with session_scope() as session:
        amount_by_service_product_id = defaultdict(float)

        q = session.query(
            Order.purchase['orders']
        ).filter(
            Order.creation_ts.isnot(None),
            Order.creation_ts >= start_range,
            Order.creation_ts <= end_range,
        )

        for (purchase_orders, ) in q:
            if purchase_orders is None:
                continue

            for purchase_order in purchase_orders:
                paid_amount = float(purchase_order['paid_amount'])
                service_product_id = purchase_order['service_product_id']

                amount_by_service_product_id[service_product_id] += paid_amount

    logger.info('Count service products %d', len(amount_by_service_product_id))
    return dict(amount_by_service_product_id)


def month_report_to_csv(report):
    return dump_csv(itertools.chain(
        (
            ['service product', 'total paid amount'],
        ),
        (
            [product, locale.format_string('%.2f', report[product])] for product in sorted(report.keys())
        )
    ))


def month_report(year, month, emails):
    logger.info('Start to create a month order report for %d year and month %d', year, month)
    start_day = 1
    end_day = calendar.monthrange(year, month)[1]

    start_range = datetime(year, month, start_day, 0, 0, 0)
    end_range = datetime(year, month, end_day, 23, 59, 59)

    report = range_report(start_range, end_range)
    csv_report = month_report_to_csv(report)

    human_month = get_human_month(month)
    mail_sender.send_email(
        to_emails=emails,
        subject='Yandex.Bus: Месячный отчет за {} месяц {} года'.format(human_month, year),
        body="""
Добрый день!

<p>Для вас был сформулирован месячный отчет об обороте сервиса Yandex.Bus за {} месяц {} года.</p>
<p>Во вложении к этому письму вы его обязательно найдете.</p>

С уважением, команда Yandex.Bus.
""".format(human_month, year),
        files=[
            (
                'month-report-{}-{:02d}.csv'.format(year, month),
                csv_report,
            ),
        ]
    )
    logger.info('Finish to create a month order report for %d year and month %d', year, month)


def year_report_to_csv(reports_by_months, max_month):
    product_to_month_data = defaultdict(dict)
    for month, report in enumerate(reports_by_months):
        for product, value in report.items():
            product_to_month_data[product][month] = value

    return dump_csv(itertools.chain(
        (
            ['service product'] + [
                get_human_month(i + 1)
                for i in range(max_month)
            ],
        ),
        (
            [product] + [
                locale.format_string('%.2f', product_to_month_data[product].get(i, 0))
                for i in range(max_month)
            ]
            for product in sorted(product_to_month_data.keys())
        )
    ))


def year_report(year, emails):
    logger.info('Start to create a month order report for %d year', year)

    max_month = 12
    now = datetime.utcnow()

    if now.year < year:
        logger.warn('Future year %d', year)
        return

    if now.year == year:
        max_month = now.month - 1
        if max_month == 0:
            logger.warn('Empty report %d year', year)
            return

    reports_by_months = []
    for month in range(1, max_month + 1):
        start_day = 1
        end_day = calendar.monthrange(year, month)[1]

        start_range = datetime(year, month, start_day, 0, 0, 0)
        end_range = datetime(year, month, end_day, 23, 59, 59)

        reports_by_months.append(range_report(start_range, end_range))

    csv_report = year_report_to_csv(reports_by_months, max_month)
    mail_sender.send_email(
        to_emails=emails,
        subject='Yandex.Bus: Годовой отчет за {} год'.format(year),
        body="""
Добрый день!

<p>Для вас был сформулирован годовой отчет об обороте сервиса Yandex.Bus за {} год.</p>
<p>Во вложении к этому письму вы его обязательно найдете.</p>

С уважением, команда Yandex.Bus.
""".format(year),
        files=[
            (
                'year-report-{}.csv'.format(year),
                csv_report,
            ),
        ]
    )
    logger.info('Finish to create a month order report for %d year', year)
